/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.rule.engine.math;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.objecthunter.exp4j.Expression;
import net.objecthunter.exp4j.ExpressionBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ConcurrentReferenceHashMap;
import org.thingsboard.common.util.DonAsynchron;
import org.thingsboard.common.util.JacksonUtil;
import org.thingsboard.rule.engine.api.RuleNode;
import org.thingsboard.rule.engine.api.TbContext;
import org.thingsboard.rule.engine.api.TbNode;
import org.thingsboard.rule.engine.api.TbNodeConfiguration;
import org.thingsboard.rule.engine.api.TbNodeException;
import org.thingsboard.rule.engine.api.util.TbNodeUtils;
import org.thingsboard.rule.engine.math.TbMathArgument;
import org.thingsboard.rule.engine.math.TbMathArgumentType;
import org.thingsboard.rule.engine.math.TbMathArgumentValue;
import org.thingsboard.rule.engine.math.TbMathNodeConfiguration;
import org.thingsboard.rule.engine.math.TbMathResult;
import org.thingsboard.rule.engine.math.TbRuleNodeMathFunctionType;
import org.thingsboard.server.common.data.StringUtils;
import org.thingsboard.server.common.data.id.EntityId;
import org.thingsboard.server.common.data.kv.BasicTsKvEntry;
import org.thingsboard.server.common.data.kv.DoubleDataEntry;
import org.thingsboard.server.common.data.kv.KvEntry;
import org.thingsboard.server.common.data.kv.TsKvEntry;
import org.thingsboard.server.common.data.plugin.ComponentType;
import org.thingsboard.server.common.msg.TbMsg;
import org.thingsboard.server.common.msg.TbMsgMetaData;

@RuleNode(type=ComponentType.ACTION, name="math function", configClazz=TbMathNodeConfiguration.class, nodeDescription="Apply math function and save the result into the message and/or database", nodeDetails="Supports math operations like: ADD, SUB, MULT, DIV, etc and functions: SIN, COS, TAN, SEC, etc. Use 'CUSTOM' operation to specify complex math expressions.<br/><br/>You may use constant, message field, metadata field, attribute, and latest time-series as an arguments values. The result of the function may be also stored to message field, metadata field, attribute or time-series value.<br/><br/>Primary use case for this rule node is to take one or more values from the database and modify them based on data from the message. For example, you may increase `totalWaterConsumption` based on the `deltaWaterConsumption` reported by device.<br/><br/>Alternative use case is the replacement of simple JS `script` nodes with more light-weight and performant implementation. For example, you may transform Fahrenheit to Celsius (C = (F - 32) / 1.8) using CUSTOM operation and expression: (x - 32) / 1.8).<br/><br/>The execution is synchronized in scope of message originator (e.g. device) and server node. If you have rule nodes in different rule chains, they will process messages from the same originator synchronously in the scope of the server node.", uiResources={"static/rulenode/rulenode-core-config.js"}, configDirective="tbActionNodeMathFunctionConfig", icon="calculate")
public class TbMathNode
implements TbNode {
    private static final Logger log = LoggerFactory.getLogger(TbMathNode.class);
    private static final ConcurrentMap<EntityId, Semaphore> semaphores = new ConcurrentReferenceHashMap();
    private final ThreadLocal<Expression> customExpression = new ThreadLocal();
    private TbMathNodeConfiguration config;
    private boolean msgBodyToJsonConversionRequired;

    public void init(TbContext ctx, TbNodeConfiguration configuration) throws TbNodeException {
        this.config = (TbMathNodeConfiguration)TbNodeUtils.convert((TbNodeConfiguration)configuration, TbMathNodeConfiguration.class);
        TbRuleNodeMathFunctionType operation = this.config.getOperation();
        int argsCount = this.config.getArguments().size();
        if (argsCount < operation.getMinArgs() || argsCount > operation.getMaxArgs()) {
            throw new RuntimeException("Args count: " + argsCount + " does not match operation: " + operation.name());
        }
        if (TbRuleNodeMathFunctionType.CUSTOM.equals((Object)operation)) {
            if (StringUtils.isBlank((String)this.config.getCustomFunction())) {
                throw new RuntimeException("Custom function is blank!");
            }
            if (this.config.getCustomFunction().length() > 256) {
                throw new RuntimeException("Custom function is too complex (length > 256)!");
            }
        }
        this.msgBodyToJsonConversionRequired = this.config.getArguments().stream().anyMatch(arg -> TbMathArgumentType.MESSAGE_BODY.equals((Object)arg.getType()));
        this.msgBodyToJsonConversionRequired = this.msgBodyToJsonConversionRequired || TbMathArgumentType.MESSAGE_BODY.equals((Object)this.config.getResult().getType());
    }

    public void onMsg(TbContext ctx, TbMsg msg) {
        Semaphore originatorSemaphore;
        EntityId originator = msg.getOriginator();
        boolean acquired = this.tryAcquire(originator, originatorSemaphore = semaphores.computeIfAbsent(originator, tmp -> new Semaphore(1, true)));
        if (!acquired) {
            ctx.tellFailure(msg, (Throwable)new RuntimeException("Failed to process message for originator synchronously"));
            return;
        }
        try {
            List<TbMathArgument> arguments = this.config.getArguments();
            Optional<ObjectNode> msgBodyOpt = this.convertMsgBodyIfRequired(msg);
            ListenableFuture argumentValues = Futures.allAsList((Iterable)arguments.stream().map(arg -> this.resolveArguments(ctx, msg, msgBodyOpt, (TbMathArgument)arg)).collect(Collectors.toList()));
            ListenableFuture resultMsgFuture = Futures.transformAsync((ListenableFuture)argumentValues, args -> this.updateMsgAndDb(ctx, msg, msgBodyOpt, this.calculateResult(ctx, msg, (List<TbMathArgumentValue>)args)), (Executor)ctx.getDbCallbackExecutor());
            DonAsynchron.withCallback((ListenableFuture)resultMsgFuture, resultMsg -> {
                try {
                    ctx.tellSuccess(resultMsg);
                }
                finally {
                    originatorSemaphore.release();
                }
            }, t -> {
                try {
                    ctx.tellFailure(msg, t);
                }
                finally {
                    originatorSemaphore.release();
                }
            }, (Executor)ctx.getDbCallbackExecutor());
        }
        catch (Throwable e) {
            originatorSemaphore.release();
            log.warn("[{}] Failed to process message: {}", new Object[]{originator, msg, e});
            throw e;
        }
    }

    private boolean tryAcquire(EntityId originator, Semaphore originatorSemaphore) {
        boolean acquired;
        try {
            acquired = originatorSemaphore.tryAcquire(20L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            acquired = false;
            log.debug("[{}] Failed to acquire semaphore", (Object)originator, (Object)e);
        }
        return acquired;
    }

    private ListenableFuture<TbMsg> updateMsgAndDb(TbContext ctx, TbMsg msg, Optional<ObjectNode> msgBodyOpt, double result) {
        TbMathResult mathResultDef = this.config.getResult();
        switch (mathResultDef.getType()) {
            case MESSAGE_BODY: {
                return Futures.immediateFuture((Object)this.addToBody(msg, mathResultDef, msgBodyOpt, result));
            }
            case MESSAGE_METADATA: {
                return Futures.immediateFuture((Object)this.addToMeta(msg, mathResultDef, result));
            }
            case ATTRIBUTE: {
                ListenableFuture<Void> attrSave = this.saveAttribute(ctx, msg, result, mathResultDef);
                return Futures.transform(attrSave, attr -> this.addToBodyAndMeta(msg, msgBodyOpt, result, mathResultDef), (Executor)ctx.getDbCallbackExecutor());
            }
            case TIME_SERIES: {
                ListenableFuture<Void> tsSave = this.saveTimeSeries(ctx, msg, result, mathResultDef);
                return Futures.transform(tsSave, ts -> this.addToBodyAndMeta(msg, msgBodyOpt, result, mathResultDef), (Executor)ctx.getDbCallbackExecutor());
            }
        }
        throw new RuntimeException("Result type is not supported: " + mathResultDef.getType() + "!");
    }

    private ListenableFuture<Void> saveTimeSeries(TbContext ctx, TbMsg msg, double result, TbMathResult mathResultDef) {
        return ctx.getTelemetryService().saveAndNotify(ctx.getTenantId(), msg.getOriginator(), (TsKvEntry)new BasicTsKvEntry(System.currentTimeMillis(), (KvEntry)new DoubleDataEntry(mathResultDef.getKey(), Double.valueOf(result))));
    }

    private ListenableFuture<Void> saveAttribute(TbContext ctx, TbMsg msg, double result, TbMathResult mathResultDef) {
        String attributeScope = this.getAttributeScope(mathResultDef.getAttributeScope());
        if (this.isIntegerResult(mathResultDef, this.config.getOperation())) {
            long value = this.toIntValue(mathResultDef, result);
            return ctx.getTelemetryService().saveAttrAndNotify(ctx.getTenantId(), msg.getOriginator(), attributeScope, mathResultDef.getKey(), value);
        }
        double value = this.toDoubleValue(mathResultDef, result);
        return ctx.getTelemetryService().saveAttrAndNotify(ctx.getTenantId(), msg.getOriginator(), attributeScope, mathResultDef.getKey(), value);
    }

    private boolean isIntegerResult(TbMathResult mathResultDef, TbRuleNodeMathFunctionType function) {
        return function.isIntegerResult() || mathResultDef.getResultValuePrecision() == 0;
    }

    private long toIntValue(TbMathResult mathResultDef, double value) {
        return (long)value;
    }

    private double toDoubleValue(TbMathResult mathResultDef, double value) {
        return BigDecimal.valueOf(value).setScale(mathResultDef.getResultValuePrecision(), RoundingMode.HALF_UP).doubleValue();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private Optional<ObjectNode> convertMsgBodyIfRequired(TbMsg msg) {
        if (!this.msgBodyToJsonConversionRequired) return Optional.empty();
        JsonNode jsonNode = JacksonUtil.toJsonNode((String)msg.getData());
        if (!jsonNode.isObject()) throw new RuntimeException("Message body is not a JSON object!");
        return Optional.of((ObjectNode)jsonNode);
    }

    private TbMsg addToBodyAndMeta(TbMsg msg, Optional<ObjectNode> msgBodyOpt, double result, TbMathResult mathResultDef) {
        TbMsg tmpMsg = msg;
        if (mathResultDef.isAddToBody()) {
            tmpMsg = this.addToBody(tmpMsg, mathResultDef, msgBodyOpt, result);
        }
        if (mathResultDef.isAddToMetadata()) {
            tmpMsg = this.addToMeta(tmpMsg, mathResultDef, result);
        }
        return tmpMsg;
    }

    private TbMsg addToBody(TbMsg msg, TbMathResult mathResultDef, Optional<ObjectNode> msgBodyOpt, double result) {
        ObjectNode body = msgBodyOpt.get();
        if (this.isIntegerResult(mathResultDef, this.config.getOperation())) {
            body.put(mathResultDef.getKey(), this.toIntValue(mathResultDef, result));
        } else {
            body.put(mathResultDef.getKey(), this.toDoubleValue(mathResultDef, result));
        }
        return TbMsg.transformMsgData((TbMsg)msg, (String)JacksonUtil.toString((Object)body));
    }

    private TbMsg addToMeta(TbMsg msg, TbMathResult mathResultDef, double result) {
        TbMsgMetaData md = msg.getMetaData();
        if (this.isIntegerResult(mathResultDef, this.config.getOperation())) {
            md.putValue(mathResultDef.getKey(), Long.toString(this.toIntValue(mathResultDef, result)));
        } else {
            md.putValue(mathResultDef.getKey(), Double.toString(this.toDoubleValue(mathResultDef, result)));
        }
        return TbMsg.transformMsg((TbMsg)msg, (TbMsgMetaData)md);
    }

    private double calculateResult(TbContext ctx, TbMsg msg, List<TbMathArgumentValue> args) {
        switch (this.config.getOperation()) {
            case ADD: {
                return this.apply(args.get(0), args.get(1), Double::sum);
            }
            case SUB: {
                return this.apply(args.get(0), args.get(1), (a, b) -> a - b);
            }
            case MULT: {
                return this.apply(args.get(0), args.get(1), (a, b) -> a * b);
            }
            case DIV: {
                return this.apply(args.get(0), args.get(1), (a, b) -> a / b);
            }
            case SIN: {
                return this.apply(args.get(0), Math::sin);
            }
            case SINH: {
                return this.apply(args.get(0), Math::sinh);
            }
            case COS: {
                return this.apply(args.get(0), Math::cos);
            }
            case COSH: {
                return this.apply(args.get(0), Math::cosh);
            }
            case TAN: {
                return this.apply(args.get(0), Math::tan);
            }
            case TANH: {
                return this.apply(args.get(0), Math::tanh);
            }
            case ACOS: {
                return this.apply(args.get(0), Math::acos);
            }
            case ASIN: {
                return this.apply(args.get(0), Math::asin);
            }
            case ATAN: {
                return this.apply(args.get(0), Math::atan);
            }
            case ATAN2: {
                return this.apply(args.get(0), args.get(1), Math::atan2);
            }
            case EXP: {
                return this.apply(args.get(0), Math::exp);
            }
            case EXPM1: {
                return this.apply(args.get(0), Math::expm1);
            }
            case SQRT: {
                return this.apply(args.get(0), Math::sqrt);
            }
            case CBRT: {
                return this.apply(args.get(0), Math::cbrt);
            }
            case GET_EXP: {
                return this.apply(args.get(0), x -> Math.getExponent(x));
            }
            case HYPOT: {
                return this.apply(args.get(0), args.get(1), Math::hypot);
            }
            case LOG: {
                return this.apply(args.get(0), Math::log);
            }
            case LOG10: {
                return this.apply(args.get(0), Math::log10);
            }
            case LOG1P: {
                return this.apply(args.get(0), Math::log1p);
            }
            case CEIL: {
                return this.apply(args.get(0), Math::ceil);
            }
            case FLOOR: {
                return this.apply(args.get(0), Math::floor);
            }
            case FLOOR_DIV: {
                return this.apply(args.get(0), args.get(1), (a, b) -> Math.floorDiv(a.longValue(), b.longValue()));
            }
            case FLOOR_MOD: {
                return this.apply(args.get(0), args.get(1), (a, b) -> Math.floorMod(a.longValue(), b.longValue()));
            }
            case ABS: {
                return this.apply(args.get(0), Math::abs);
            }
            case MIN: {
                return this.apply(args.get(0), args.get(1), Math::min);
            }
            case MAX: {
                return this.apply(args.get(0), args.get(1), Math::max);
            }
            case POW: {
                return this.apply(args.get(0), args.get(1), Math::pow);
            }
            case SIGNUM: {
                return this.apply(args.get(0), Math::signum);
            }
            case RAD: {
                return this.apply(args.get(0), Math::toRadians);
            }
            case DEG: {
                return this.apply(args.get(0), Math::toDegrees);
            }
            case CUSTOM: {
                Expression expr = this.customExpression.get();
                if (expr == null) {
                    expr = new ExpressionBuilder(this.config.getCustomFunction()).implicitMultiplication(true).variables(this.config.getArguments().stream().map(TbMathArgument::getName).collect(Collectors.toSet())).build();
                    this.customExpression.set(expr);
                }
                for (int i = 0; i < this.config.getArguments().size(); ++i) {
                    expr.setVariable(this.config.getArguments().get(i).getName(), args.get(i).getValue());
                }
                return expr.evaluate();
            }
        }
        throw new RuntimeException("Not supported operation: " + this.config.getOperation());
    }

    private double apply(TbMathArgumentValue arg, Function<Double, Double> function) {
        return function.apply(arg.getValue());
    }

    private double apply(TbMathArgumentValue arg1, TbMathArgumentValue arg2, BiFunction<Double, Double, Double> function) {
        return function.apply(arg1.getValue(), arg2.getValue());
    }

    private ListenableFuture<TbMathArgumentValue> resolveArguments(TbContext ctx, TbMsg msg, Optional<ObjectNode> msgBodyOpt, TbMathArgument arg) {
        switch (arg.getType()) {
            case CONSTANT: {
                return Futures.immediateFuture((Object)TbMathArgumentValue.constant(arg));
            }
            case MESSAGE_BODY: {
                return Futures.immediateFuture((Object)TbMathArgumentValue.fromMessageBody(arg, msgBodyOpt));
            }
            case MESSAGE_METADATA: {
                return Futures.immediateFuture((Object)TbMathArgumentValue.fromMessageMetadata(arg, msg.getMetaData()));
            }
            case ATTRIBUTE: {
                String scope = this.getAttributeScope(arg.getAttributeScope());
                return Futures.transform((ListenableFuture)ctx.getAttributesService().find(ctx.getTenantId(), msg.getOriginator(), scope, arg.getKey()), opt -> this.getTbMathArgumentValue(arg, (Optional<? extends KvEntry>)opt, "Attribute: " + arg.getKey() + " with scope: " + scope + " not found for entity: " + msg.getOriginator()), (Executor)MoreExecutors.directExecutor());
            }
            case TIME_SERIES: {
                return Futures.transform((ListenableFuture)ctx.getTimeseriesService().findLatest(ctx.getTenantId(), msg.getOriginator(), arg.getKey()), opt -> this.getTbMathArgumentValue(arg, (Optional<? extends KvEntry>)opt, "Time-series: " + arg.getKey() + " not found for entity: " + msg.getOriginator()), (Executor)MoreExecutors.directExecutor());
            }
        }
        throw new RuntimeException("Unsupported argument type: " + arg.getType() + "!");
    }

    private String getAttributeScope(String attrScope) {
        return StringUtils.isEmpty((String)attrScope) ? "SERVER_SCOPE" : attrScope;
    }

    private TbMathArgumentValue getTbMathArgumentValue(TbMathArgument arg, Optional<? extends KvEntry> kvOpt, String error) {
        if (kvOpt != null && kvOpt.isPresent()) {
            KvEntry kv = kvOpt.get();
            switch (kv.getDataType()) {
                case LONG: {
                    return TbMathArgumentValue.fromLong((Long)kv.getLongValue().get());
                }
                case DOUBLE: {
                    return TbMathArgumentValue.fromDouble((Double)kv.getDoubleValue().get());
                }
            }
            return TbMathArgumentValue.fromString(kv.getValueAsString());
        }
        if (arg.getDefaultValue() != null) {
            return TbMathArgumentValue.fromDouble(arg.getDefaultValue());
        }
        throw new RuntimeException(error);
    }

    public void destroy() {
    }
}

