/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.migrator.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.annotation.PostConstruct;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.postgresql.jdbc.PgArray;
import org.postgresql.util.PGobject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.thingsboard.migrator.utils.PostgresService;

@Component
public class Storage {
    private static final Logger log = LoggerFactory.getLogger(Storage.class);
    @Value(value="${working_directory}")
    private Path workingDir;
    @Value(value="${mode}")
    private String mode;
    private static final String FINAL_ARCHIVE_FILE = "data.tar";
    private final ObjectMapper jsonMapper = new ObjectMapper();
    private final List<Path> createdFiles = new ArrayList();
    private static final String BYTES_DATA_PREFIX = "BYTES:";
    private static final String ARRAY_DATA_PREFIX = "ARRAY:";
    private static final String BLOB_DATA_PREFIX = "BLOB:";

    @PostConstruct
    private void init() throws IOException {
        Files.createDirectories(this.workingDir, new FileAttribute[0]);
    }

    public void newFile(String name) throws IOException {
        Path file = this.getPath(name);
        Files.deleteIfExists(file);
        Files.createFile(file, new FileAttribute[0]);
        this.createdFiles.add(file);
    }

    public Writer newWriter(String file) {
        FileOutputStream fileOutputStream = new FileOutputStream(this.getPath(file).toFile());
        return new OutputStreamWriter((OutputStream)new GZIPOutputStream(fileOutputStream), StandardCharsets.UTF_8);
    }

    public void addToFile(Writer writer, Map<String, Object> row) {
        row.replaceAll((column, data) -> {
            if (data instanceof PGobject) {
                PGobject object = (PGobject)data;
                data = object.getValue();
            } else if (data instanceof byte[]) {
                byte[] bytes = (byte[])data;
                data = BYTES_DATA_PREFIX + Base64.getEncoder().encodeToString(bytes);
            } else {
                if (data instanceof PgArray) {
                    PgArray array = (PgArray)data;
                    try {
                        return ARRAY_DATA_PREFIX + this.jsonMapper.writeValueAsString(array.getArray());
                    }
                    catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
                if (data instanceof PostgresService.Blob) {
                    PostgresService.Blob blob = (PostgresService.Blob)data;
                    return BLOB_DATA_PREFIX + Base64.getEncoder().encodeToString(blob.data());
                }
            }
            return data;
        });
        String serialized = this.toJson(row);
        writer.write(serialized + System.lineSeparator());
    }

    public void readAndProcess(String file, Consumer<Map<String, Object>> processor) throws IOException {
        try (BufferedReader reader = this.newReader(file);){
            reader.lines().forEach(line -> {
                if (StringUtils.isNotBlank((CharSequence)line)) {
                    Map data;
                    try {
                        data = (Map)this.jsonMapper.readValue(line, (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
                    }
                    catch (JsonProcessingException e) {
                        throw new RuntimeException(e);
                    }
                    data.replaceAll((key, value) -> {
                        if (value == null) {
                            return null;
                        }
                        if (key.contains("id")) {
                            try {
                                value = UUID.fromString(value.toString());
                            }
                            catch (IllegalArgumentException illegalArgumentException) {}
                        } else if (value instanceof Map) {
                            value = this.jsonMapper.valueToTree(value);
                        } else if (value instanceof String) {
                            String string = (String)value;
                            if (string.startsWith(BYTES_DATA_PREFIX)) {
                                value = Base64.getDecoder().decode(StringUtils.removeStart((String)string, (String)BYTES_DATA_PREFIX));
                            } else if (string.startsWith(ARRAY_DATA_PREFIX)) {
                                try {
                                    value = this.jsonMapper.readValue(StringUtils.removeStart((String)string, (String)ARRAY_DATA_PREFIX), (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
                                }
                                catch (JsonProcessingException e) {
                                    throw new RuntimeException(e);
                                }
                            } else if (string.startsWith(BLOB_DATA_PREFIX)) {
                                byte[] blobData = Base64.getDecoder().decode(StringUtils.removeStart((String)string, (String)BLOB_DATA_PREFIX));
                                return new PostgresService.Blob(blobData);
                            }
                        }
                        return value;
                    });
                    processor.accept(data);
                }
            });
        }
    }

    private BufferedReader newReader(String file) throws IOException {
        FileInputStream fileInputStream = new FileInputStream(this.getPath(file).toFile());
        return new BufferedReader(new InputStreamReader(new GZIPInputStream(fileInputStream)));
    }

    private String toJson(Object o) {
        return this.jsonMapper.writeValueAsString(o);
    }

    private Path getPath(String file) {
        return this.workingDir.resolve(file + ".gz");
    }

    public void close() {
        if (this.createdFiles.isEmpty()) {
            return;
        }
        if (this.mode.equals("TENANT_DATA_EXPORT")) {
            log.info("Archiving {} files", (Object)this.createdFiles.size());
            TarArchiveOutputStream tarArchive = new TarArchiveOutputStream((OutputStream)new FileOutputStream(this.workingDir.resolve(FINAL_ARCHIVE_FILE).toFile()));
            for (Path file : this.createdFiles) {
                TarArchiveEntry archiveEntry = new TarArchiveEntry(file, file.getFileName().toString(), new LinkOption[0]);
                tarArchive.putArchiveEntry(archiveEntry);
                Files.copy(file, (OutputStream)tarArchive);
                tarArchive.closeArchiveEntry();
                Files.delete(file);
            }
            tarArchive.close();
        }
    }

    public void open() {
        if (this.mode.equals("TENANT_DATA_IMPORT")) {
            Stream<Path> archives = Files.list(this.workingDir);
            archives.filter(file -> file.getFileName().endsWith(".tar")).forEach(archiveFile -> {
                try (TarArchiveInputStream tarArchive = new TarArchiveInputStream((InputStream)new FileInputStream(archiveFile.toFile()));){
                    TarArchiveEntry entry;
                    log.info("Unarchiving {}", (Object)archiveFile.getFileName());
                    while ((entry = tarArchive.getNextEntry()) != null) {
                        try (FileOutputStream file = new FileOutputStream(this.workingDir.resolve(entry.getName()).toFile());){
                            tarArchive.transferTo((OutputStream)file);
                        }
                    }
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            archives.close();
        }
    }
}

