/*
 * Decompiled with CFR 0.152.
 */
package dev.langchain4j.model.googleai;

import dev.langchain4j.exception.HttpException;
import dev.langchain4j.model.googleai.GeminiCachedContent;
import dev.langchain4j.model.googleai.GeminiContent;
import dev.langchain4j.model.googleai.GeminiPart;
import dev.langchain4j.model.googleai.GeminiService;
import dev.langchain4j.model.googleai.GoogleAiListCachedContentsRequest;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GeminiCacheManager {
    private static final Logger log = LoggerFactory.getLogger(GeminiCacheManager.class);
    private final GeminiService geminiService;
    private final ConcurrentMap<String, CachedContentMetadata> cachedContents;

    public GeminiCacheManager(GeminiService geminiService) {
        this.geminiService = geminiService;
        GoogleAiListCachedContentsRequest listCachedContentsRequest = new GoogleAiListCachedContentsRequest();
        listCachedContentsRequest.setPageSize(1000);
        this.cachedContents = new ConcurrentHashMap<String, CachedContentMetadata>((Map)Optional.ofNullable(geminiService.listCachedContents(listCachedContentsRequest).getCachedContents()).orElse(Collections.emptyList()).stream().map(CachedContentMetadata::new).collect(Collectors.toConcurrentMap(CachedContentMetadata::getKey, Function.identity(), BinaryOperator.maxBy(Comparator.comparing(CachedContentMetadata::getExpirationTime)))));
        log.debug("Loaded existing cached contents: {}", this.cachedContents);
    }

    public String getOrCreateCached(String key, Duration ttl, GeminiContent content, String model) {
        return this.cachedContents.compute(key, (__, cachedContent) -> {
            if (cachedContent != null) {
                if (cachedContent.isExpired()) {
                    log.debug("Cached content for key '{}' is expired: {}", (Object)key, cachedContent);
                } else if (cachedContent.isAlmostExpired()) {
                    log.debug("Cached content for key '{}' is almost expired: {}", (Object)key, cachedContent);
                    this.deleteCachedContent((CachedContentMetadata)cachedContent);
                } else {
                    if (cachedContent.isChecksumVerified()) {
                        log.debug("Using existing cached content for key '{}': {}", (Object)key, cachedContent);
                        return cachedContent;
                    }
                    if (cachedContent.checksumMatches(content)) {
                        cachedContent.setChecksumVerified(true);
                        log.debug("Using existing cached content for key '{}' with matching checksum: {}", (Object)key, cachedContent);
                        return cachedContent;
                    }
                    log.debug("Cached content for key '{}' has different checksum: {}", (Object)key, cachedContent);
                    this.deleteCachedContent((CachedContentMetadata)cachedContent);
                }
            }
            return this.createCachedContent(key, ttl, content, model);
        }).getId();
    }

    private void deleteCachedContent(CachedContentMetadata cachedContent) {
        try {
            log.debug("Deleting cached content for key '{}': {}", (Object)cachedContent.getKey(), (Object)cachedContent);
            this.geminiService.deleteCachedContent(StringUtils.removeStart((String)cachedContent.getId(), (String)"cachedContents/"));
        }
        catch (HttpException e) {
            if (e.statusCode() == 403 || e.statusCode() == 404) {
                log.debug("Couldn't delete cached content for key '{}': {}", (Object)cachedContent.getKey(), (Object)e.getMessage());
            }
            throw e;
        }
    }

    private CachedContentMetadata createCachedContent(String key, Duration ttl, GeminiContent content, String model) {
        GeminiCachedContent cachedContent = GeminiCachedContent.builder().systemInstruction(content).ttl(ttl.toSeconds() + "s").displayName(key + ":" + GeminiCacheManager.getChecksum(content)).build();
        cachedContent = this.geminiService.createCachedContent(model, cachedContent);
        CachedContentMetadata newCachedContent = new CachedContentMetadata(cachedContent);
        newCachedContent.setChecksumVerified(true);
        log.debug("Created new cached content for key '{}': {}", (Object)key, (Object)cachedContent);
        return newCachedContent;
    }

    private static String getChecksum(GeminiContent content) {
        return DigestUtils.sha256Hex((String)content.getParts().stream().map(GeminiPart::getText).collect(Collectors.joining(System.lineSeparator())));
    }

    private static class CachedContentMetadata {
        String id;
        String key;
        String checksum;
        Instant expirationTime;
        GeminiCachedContent cachedContent;
        boolean checksumVerified;

        CachedContentMetadata(GeminiCachedContent cachedContent) {
            this.id = cachedContent.getName();
            String[] parts = cachedContent.getDisplayName().split(":");
            this.key = parts[0];
            this.checksum = parts.length == 2 ? parts[1] : "undefined";
            this.expirationTime = Instant.parse(cachedContent.getExpireTime());
            this.cachedContent = cachedContent;
            this.checksumVerified = false;
        }

        public String getId() {
            return this.id;
        }

        public String getKey() {
            return this.key;
        }

        public String getChecksum() {
            return this.checksum;
        }

        public boolean isChecksumVerified() {
            return this.checksumVerified;
        }

        public void setChecksumVerified(boolean checksumVerified) {
            this.checksumVerified = checksumVerified;
        }

        public Instant getExpirationTime() {
            return this.expirationTime;
        }

        public boolean isAlmostExpired() {
            return this.expirationTime.minusSeconds(60L).isBefore(Instant.now());
        }

        public boolean isExpired() {
            return this.expirationTime.isBefore(Instant.now());
        }

        public boolean checksumMatches(GeminiContent content) {
            return this.checksum.equals(GeminiCacheManager.getChecksum(content));
        }

        public String toString() {
            return "CachedContentMetadata{id='" + this.id + "', key='" + this.key + "', checksum='" + this.checksum + "', expirationTime=" + String.valueOf(this.expirationTime) + ", checksumVerified=" + this.checksumVerified + "}";
        }
    }
}

