/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.server.dao.service;

import com.datastax.oss.driver.api.core.uuid.Uuids;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.thingsboard.server.common.data.BaseData;
import org.thingsboard.server.common.data.Customer;
import org.thingsboard.server.common.data.Device;
import org.thingsboard.server.common.data.DeviceInfo;
import org.thingsboard.server.common.data.DeviceInfoFilter;
import org.thingsboard.server.common.data.DeviceProfile;
import org.thingsboard.server.common.data.EntitySubtype;
import org.thingsboard.server.common.data.HasOtaPackage;
import org.thingsboard.server.common.data.OtaPackage;
import org.thingsboard.server.common.data.OtaPackageInfo;
import org.thingsboard.server.common.data.StringUtils;
import org.thingsboard.server.common.data.Tenant;
import org.thingsboard.server.common.data.TenantProfile;
import org.thingsboard.server.common.data.cf.CalculatedField;
import org.thingsboard.server.common.data.cf.CalculatedFieldType;
import org.thingsboard.server.common.data.cf.configuration.Argument;
import org.thingsboard.server.common.data.cf.configuration.ArgumentType;
import org.thingsboard.server.common.data.cf.configuration.CalculatedFieldConfiguration;
import org.thingsboard.server.common.data.cf.configuration.Output;
import org.thingsboard.server.common.data.cf.configuration.OutputType;
import org.thingsboard.server.common.data.cf.configuration.ReferencedEntityKey;
import org.thingsboard.server.common.data.cf.configuration.SimpleCalculatedFieldConfiguration;
import org.thingsboard.server.common.data.id.CustomerId;
import org.thingsboard.server.common.data.id.DeviceProfileId;
import org.thingsboard.server.common.data.id.EntityId;
import org.thingsboard.server.common.data.id.OtaPackageId;
import org.thingsboard.server.common.data.id.TenantId;
import org.thingsboard.server.common.data.ota.ChecksumAlgorithm;
import org.thingsboard.server.common.data.ota.OtaPackageType;
import org.thingsboard.server.common.data.page.PageData;
import org.thingsboard.server.common.data.page.PageLink;
import org.thingsboard.server.common.data.security.DeviceCredentials;
import org.thingsboard.server.common.data.security.DeviceCredentialsType;
import org.thingsboard.server.common.data.tenant.profile.DefaultTenantProfileConfiguration;
import org.thingsboard.server.common.data.tenant.profile.TenantProfileConfiguration;
import org.thingsboard.server.dao.cf.CalculatedFieldService;
import org.thingsboard.server.dao.customer.CustomerService;
import org.thingsboard.server.dao.device.DeviceCredentialsService;
import org.thingsboard.server.dao.device.DeviceProfileService;
import org.thingsboard.server.dao.device.DeviceService;
import org.thingsboard.server.dao.exception.DataValidationException;
import org.thingsboard.server.dao.exception.DeviceCredentialsValidationException;
import org.thingsboard.server.dao.model.ModelConstants;
import org.thingsboard.server.dao.ota.OtaPackageService;
import org.thingsboard.server.dao.service.AbstractServiceTest;
import org.thingsboard.server.dao.service.DaoSqlTest;
import org.thingsboard.server.dao.service.validator.DeviceCredentialsDataValidator;
import org.thingsboard.server.dao.tenant.TenantProfileService;

@DaoSqlTest
public class DeviceServiceTest
extends AbstractServiceTest {
    @Autowired
    CustomerService customerService;
    @Autowired
    DeviceCredentialsService deviceCredentialsService;
    @Autowired
    DeviceProfileService deviceProfileService;
    @Autowired
    DeviceService deviceService;
    @Autowired
    OtaPackageService otaPackageService;
    @Autowired
    TenantProfileService tenantProfileService;
    @Autowired
    private CalculatedFieldService calculatedFieldService;
    @Autowired
    private PlatformTransactionManager platformTransactionManager;
    @SpyBean
    private DeviceCredentialsDataValidator validator;
    private AbstractServiceTest.IdComparator<Device> idComparator = new AbstractServiceTest.IdComparator();
    private TenantId anotherTenantId;

    @Before
    public void before() {
        this.anotherTenantId = this.createTenant().getId();
    }

    @After
    public void after() {
        this.tenantService.deleteTenant(this.tenantId);
        this.tenantService.deleteTenant(this.anotherTenantId);
        this.tenantProfileService.deleteTenantProfiles(this.tenantId);
        this.tenantProfileService.deleteTenantProfiles(this.anotherTenantId);
    }

    @Test
    public void testSaveDevicesWithoutMaxDeviceLimit() {
        Device device = this.saveDevice(this.tenantId, "My device");
        this.deleteDevice(this.tenantId, device);
    }

    @Test
    public void testSaveDevicesWithInfiniteMaxDeviceLimit() {
        TenantProfile defaultTenantProfile = this.tenantProfileService.findDefaultTenantProfile(this.tenantId);
        defaultTenantProfile.getProfileData().setConfiguration((TenantProfileConfiguration)DefaultTenantProfileConfiguration.builder().maxDevices(Long.MAX_VALUE).build());
        this.tenantProfileService.saveTenantProfile(this.tenantId, defaultTenantProfile);
        Device device = this.saveDevice(this.tenantId, "My device");
        this.deleteDevice(this.tenantId, device);
    }

    @Test
    public void testSaveDevicesWithMaxDeviceOutOfLimit() {
        TenantProfile defaultTenantProfile = this.tenantProfileService.findDefaultTenantProfile(this.tenantId);
        defaultTenantProfile.getProfileData().setConfiguration((TenantProfileConfiguration)DefaultTenantProfileConfiguration.builder().maxDevices(1L).build());
        this.tenantProfileService.saveTenantProfile(this.tenantId, defaultTenantProfile);
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(this.tenantId));
        this.saveDevice(this.tenantId, "My first device");
        Assert.assertEquals((long)1L, (long)this.deviceService.countByTenantId(this.tenantId));
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.saveDevice(this.tenantId, "My second device that out of maxDeviceCount limit"));
    }

    @Test
    public void testSaveDevicesWithTheSameAccessToken() {
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName(StringUtils.randomAlphabetic((int)10));
        device.setType("default");
        String accessToken = StringUtils.generateSafeToken((int)10);
        Device savedDevice = this.deviceService.saveDeviceWithAccessToken(device, accessToken);
        DeviceCredentials deviceCredentials = this.deviceCredentialsService.findDeviceCredentialsByDeviceId(this.tenantId, savedDevice.getId());
        Assert.assertEquals((Object)accessToken, (Object)deviceCredentials.getCredentialsId());
        Device duplicatedDevice = new Device();
        duplicatedDevice.setTenantId(this.tenantId);
        duplicatedDevice.setName(StringUtils.randomAlphabetic((int)10));
        duplicatedDevice.setType("default");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.deviceService.saveDeviceWithAccessToken(duplicatedDevice, accessToken)).isInstanceOf(DeviceCredentialsValidationException.class)).hasMessageContaining("Device credentials are already assigned to another device!");
        Device deviceByName = this.deviceService.findDeviceByTenantIdAndName(this.tenantId, duplicatedDevice.getName());
        org.junit.jupiter.api.Assertions.assertNull((Object)deviceByName);
    }

    @Test
    public void testShouldRollbackValidatedDeviceIfDeviceCredentialsValidationFailed() {
        Mockito.reset((Object[])new DeviceCredentialsDataValidator[]{this.validator});
        ((DeviceCredentialsDataValidator)Mockito.doThrow((Throwable[])new Throwable[]{new DataValidationException("mock message")}).when((Object)this.validator)).validate((BaseData)((DeviceCredentials)ArgumentMatchers.any()), (Function)ArgumentMatchers.any());
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName(StringUtils.randomAlphabetic((int)10));
        device.setType("default");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.deviceService.saveDevice(device)).isInstanceOf(DataValidationException.class)).hasMessageContaining("mock message");
        Device deviceByName = this.deviceService.findDeviceByTenantIdAndName(this.tenantId, device.getName());
        org.junit.jupiter.api.Assertions.assertNull((Object)deviceByName);
    }

    @Test
    public void testCountByTenantId() {
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(this.tenantId));
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(this.anotherTenantId));
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(TenantId.SYS_TENANT_ID));
        Device anotherDevice = this.saveDevice(this.anotherTenantId, "My device 1");
        Assert.assertEquals((long)1L, (long)this.deviceService.countByTenantId(this.anotherTenantId));
        int maxDevices = 8;
        ArrayList<Device> devices = new ArrayList<Device>(maxDevices);
        for (int i = 1; i <= maxDevices; ++i) {
            devices.add(this.saveDevice(this.tenantId, "My device " + i));
            Assert.assertEquals((long)i, (long)this.deviceService.countByTenantId(this.tenantId));
        }
        Assert.assertEquals((long)maxDevices, (long)this.deviceService.countByTenantId(this.tenantId));
        Assert.assertEquals((long)1L, (long)this.deviceService.countByTenantId(this.anotherTenantId));
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(TenantId.SYS_TENANT_ID));
        devices.forEach(device -> this.deleteDevice(this.tenantId, (Device)device));
        this.deleteDevice(this.anotherTenantId, anotherDevice);
    }

    @Test
    public void testCountDevicesWithoutFirmware() {
        this.testCountDevicesWithoutOta(OtaPackageType.FIRMWARE);
    }

    @Test
    public void testCountDevicesWithoutSoftware() {
        this.testCountDevicesWithoutOta(OtaPackageType.SOFTWARE);
    }

    public void testCountDevicesWithoutOta(OtaPackageType type) {
        DeviceProfile defaultDeviceProfile = this.deviceProfileService.findDefaultDeviceProfile(this.tenantId);
        DeviceProfileId deviceProfileId = defaultDeviceProfile.getId();
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(this.tenantId));
        Assert.assertEquals((long)0L, (long)this.deviceService.countDevicesByTenantIdAndDeviceProfileIdAndEmptyOtaPackage(this.tenantId, deviceProfileId, type));
        int maxDevices = 8;
        ArrayList<Device> devices = new ArrayList<Device>(maxDevices);
        for (int i = 1; i <= maxDevices; ++i) {
            devices.add(this.saveDevice(this.tenantId, "My device " + i));
            Assert.assertEquals((long)i, (long)this.deviceService.countDevicesByTenantIdAndDeviceProfileIdAndEmptyOtaPackage(this.tenantId, deviceProfileId, type));
        }
        Assert.assertEquals((long)maxDevices, (long)this.deviceService.countDevicesByTenantIdAndDeviceProfileIdAndEmptyOtaPackage(this.tenantId, deviceProfileId, type));
        OtaPackageId otaPackageId = this.createOta(deviceProfileId, type);
        int devicesWithOta = maxDevices / 2;
        for (int i = 0; i < devicesWithOta; ++i) {
            Device device2 = (Device)devices.get(i);
            this.setOtaPackageId(device2, type, otaPackageId);
            this.deviceService.saveDevice(device2);
        }
        Assert.assertEquals((long)(maxDevices - devicesWithOta), (long)this.deviceService.countDevicesByTenantIdAndDeviceProfileIdAndEmptyOtaPackage(this.tenantId, deviceProfileId, type));
        devices.forEach(device -> this.deleteDevice(this.tenantId, (Device)device));
    }

    @Test
    public void testFindDevicesWithoutFirmware() {
        this.testFindDevicesWithoutOta(OtaPackageType.FIRMWARE);
    }

    @Test
    public void testFindDevicesWithoutSoftware() {
        this.testFindDevicesWithoutOta(OtaPackageType.SOFTWARE);
    }

    public void testFindDevicesWithoutOta(OtaPackageType type) {
        int i;
        DeviceProfile defaultDeviceProfile = this.deviceProfileService.findDefaultDeviceProfile(this.tenantId);
        DeviceProfileId deviceProfileId = defaultDeviceProfile.getId();
        PageLink pageLink = new PageLink(100);
        Assert.assertEquals((long)0L, (long)this.deviceService.countByTenantId(this.tenantId));
        Assert.assertEquals((long)0L, (long)this.deviceService.findDevicesByTenantIdAndTypeAndEmptyOtaPackage(this.tenantId, deviceProfileId, type, pageLink).getData().size());
        int maxDevices = 8;
        ArrayList<Device> devices = new ArrayList<Device>(maxDevices);
        for (int i2 = 1; i2 <= maxDevices; ++i2) {
            devices.add(this.saveDevice(this.tenantId, "My device " + i2));
        }
        List foundDevices = this.deviceService.findDevicesByTenantIdAndTypeAndEmptyOtaPackage(this.tenantId, deviceProfileId, type, pageLink).getData();
        Assert.assertEquals((long)maxDevices, (long)foundDevices.size());
        devices.sort(this.idComparator);
        foundDevices.sort(this.idComparator);
        Assert.assertEquals(devices, (Object)foundDevices);
        OtaPackageId otaPackageId = this.createOta(deviceProfileId, type);
        int devicesWithOta = maxDevices / 2;
        for (i = 0; i < devicesWithOta; ++i) {
            Device device2 = (Device)devices.get(i);
            this.setOtaPackageId(device2, type, otaPackageId);
            this.deviceService.saveDevice(device2);
        }
        foundDevices = this.deviceService.findDevicesByTenantIdAndTypeAndEmptyOtaPackage(this.tenantId, deviceProfileId, type, pageLink).getData();
        Assert.assertEquals((long)(maxDevices - devicesWithOta), (long)foundDevices.size());
        foundDevices.sort(this.idComparator);
        for (i = 0; i < foundDevices.size(); ++i) {
            Assert.assertEquals(devices.get(i + devicesWithOta), foundDevices.get(i));
        }
        devices.forEach(device -> this.deleteDevice(this.tenantId, (Device)device));
    }

    private <T extends HasOtaPackage> void setOtaPackageId(T obj, OtaPackageType type, OtaPackageId otaPackageId) {
        switch (type) {
            case FIRMWARE: {
                obj.setFirmwareId(otaPackageId);
                break;
            }
            case SOFTWARE: {
                obj.setSoftwareId(otaPackageId);
            }
        }
    }

    private OtaPackageId createOta(DeviceProfileId deviceProfileId, OtaPackageType type) {
        OtaPackageInfo ota = new OtaPackageInfo();
        ota.setTenantId(this.tenantId);
        ota.setDeviceProfileId(deviceProfileId);
        ota.setType(type);
        ota.setTitle("Test_" + String.valueOf(type));
        ota.setVersion("v1.0");
        ota.setUrl("http://ota.test.org");
        ota.setDataSize(Long.valueOf(0L));
        OtaPackageInfo savedOta = this.otaPackageService.saveOtaPackageInfo(ota, true);
        Assert.assertNotNull((Object)savedOta);
        return savedOta.getId();
    }

    void deleteDevice(TenantId tenantId, Device device) {
        this.deviceService.deleteDevice(tenantId, device.getId());
    }

    Device saveDevice(TenantId tenantId, String name) {
        Device device = new Device();
        device.setTenantId(tenantId);
        device.setName(name);
        device.setType("default");
        Device savedDevice = this.deviceService.saveDevice(device);
        Assert.assertNotNull((Object)savedDevice);
        Assert.assertNotNull((Object)savedDevice.getId());
        Assert.assertTrue((savedDevice.getCreatedTime() > 0L ? 1 : 0) != 0);
        Assert.assertEquals((Object)device.getTenantId(), (Object)savedDevice.getTenantId());
        Assert.assertNotNull((Object)savedDevice.getCustomerId());
        Assert.assertEquals((Object)ModelConstants.NULL_UUID, (Object)savedDevice.getCustomerId().getId());
        Assert.assertEquals((Object)device.getName(), (Object)savedDevice.getName());
        DeviceCredentials deviceCredentials = this.deviceCredentialsService.findDeviceCredentialsByDeviceId(tenantId, savedDevice.getId());
        Assert.assertNotNull((Object)deviceCredentials);
        Assert.assertNotNull((Object)deviceCredentials.getId());
        Assert.assertEquals((Object)savedDevice.getId(), (Object)deviceCredentials.getDeviceId());
        Assert.assertEquals((Object)DeviceCredentialsType.ACCESS_TOKEN, (Object)deviceCredentials.getCredentialsType());
        Assert.assertNotNull((Object)deviceCredentials.getCredentialsId());
        Assert.assertEquals((long)20L, (long)deviceCredentials.getCredentialsId().length());
        savedDevice.setName("New " + savedDevice.getName());
        this.deviceService.saveDevice(savedDevice);
        Device foundDevice = this.deviceService.findDeviceById(tenantId, savedDevice.getId());
        Assert.assertEquals((Object)foundDevice.getName(), (Object)savedDevice.getName());
        return foundDevice;
    }

    @Test
    public void testSaveDeviceWithFirmware() {
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("My device");
        device.setType("default");
        Device savedDevice = this.deviceService.saveDevice(device);
        Assert.assertNotNull((Object)savedDevice);
        Assert.assertNotNull((Object)savedDevice.getId());
        Assert.assertTrue((savedDevice.getCreatedTime() > 0L ? 1 : 0) != 0);
        Assert.assertEquals((Object)device.getTenantId(), (Object)savedDevice.getTenantId());
        Assert.assertNotNull((Object)savedDevice.getCustomerId());
        Assert.assertEquals((Object)ModelConstants.NULL_UUID, (Object)savedDevice.getCustomerId().getId());
        Assert.assertEquals((Object)device.getName(), (Object)savedDevice.getName());
        DeviceCredentials deviceCredentials = this.deviceCredentialsService.findDeviceCredentialsByDeviceId(this.tenantId, savedDevice.getId());
        Assert.assertNotNull((Object)deviceCredentials);
        Assert.assertNotNull((Object)deviceCredentials.getId());
        Assert.assertEquals((Object)savedDevice.getId(), (Object)deviceCredentials.getDeviceId());
        Assert.assertEquals((Object)DeviceCredentialsType.ACCESS_TOKEN, (Object)deviceCredentials.getCredentialsType());
        Assert.assertNotNull((Object)deviceCredentials.getCredentialsId());
        Assert.assertEquals((long)20L, (long)deviceCredentials.getCredentialsId().length());
        OtaPackage firmware = new OtaPackage();
        firmware.setTenantId(this.tenantId);
        firmware.setDeviceProfileId(device.getDeviceProfileId());
        firmware.setType(OtaPackageType.FIRMWARE);
        firmware.setTitle("my firmware");
        firmware.setVersion("v1.0");
        firmware.setFileName("test.txt");
        firmware.setContentType("text/plain");
        firmware.setChecksumAlgorithm(ChecksumAlgorithm.SHA256);
        firmware.setChecksum("4bf5122f344554c53bde2ebb8cd2b7e3d1600ad631c385a5d7cce23c7785459a");
        firmware.setData(ByteBuffer.wrap(new byte[]{1}));
        firmware.setDataSize(Long.valueOf(1L));
        OtaPackage savedFirmware = this.otaPackageService.saveOtaPackage(firmware);
        savedDevice.setFirmwareId(savedFirmware.getId());
        this.deviceService.saveDevice(savedDevice);
        Device foundDevice = this.deviceService.findDeviceById(this.tenantId, savedDevice.getId());
        Assert.assertEquals((Object)foundDevice.getName(), (Object)savedDevice.getName());
    }

    @Test
    public void testAssignFirmwareToDeviceWithDifferentDeviceProfile() {
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("My device");
        device.setType("default");
        Device savedDevice = this.deviceService.saveDevice(device);
        Assert.assertNotNull((Object)savedDevice);
        DeviceProfile deviceProfile = this.createDeviceProfile(this.tenantId, "New device Profile");
        DeviceProfile savedProfile = this.deviceProfileService.saveDeviceProfile(deviceProfile);
        Assert.assertNotNull((Object)savedProfile);
        OtaPackage firmware = new OtaPackage();
        firmware.setTenantId(this.tenantId);
        firmware.setDeviceProfileId(savedProfile.getId());
        firmware.setType(OtaPackageType.FIRMWARE);
        firmware.setTitle("my firmware");
        firmware.setVersion("v1.0");
        firmware.setFileName("test.txt");
        firmware.setContentType("text/plain");
        firmware.setChecksumAlgorithm(ChecksumAlgorithm.SHA256);
        firmware.setChecksum("4bf5122f344554c53bde2ebb8cd2b7e3d1600ad631c385a5d7cce23c7785459a");
        firmware.setData(ByteBuffer.wrap(new byte[]{1}));
        firmware.setDataSize(Long.valueOf(1L));
        OtaPackage savedFirmware = this.otaPackageService.saveOtaPackage(firmware);
        savedDevice.setFirmwareId(savedFirmware.getId());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.deviceService.saveDevice(savedDevice)).isInstanceOf(DataValidationException.class)).hasMessageContaining("Can't assign firmware with different deviceProfile!");
    }

    @Test
    public void testSaveDeviceWithEmptyName() {
        Device device = new Device();
        device.setType("default");
        device.setTenantId(this.tenantId);
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.saveDevice(device));
    }

    @Test
    public void testSaveDeviceWithEmptyTenant() {
        Device device = new Device();
        device.setName("My device");
        device.setType("default");
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.saveDevice(device));
    }

    @Test
    public void testSaveDeviceWithNameContains0x00_thenDataValidationException() {
        Device device = new Device();
        device.setType("default");
        device.setTenantId(this.tenantId);
        device.setName("F0929906\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000");
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.saveDevice(device));
    }

    @Test
    public void testSaveDeviceWithJSInjection_thenDataValidationException() {
        Device device = new Device();
        device.setType("default");
        device.setTenantId(this.tenantId);
        device.setName("{{constructor.constructor('location.href=\"https://evil.com\"')()}}");
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.saveDevice(device));
    }

    @Test
    public void testSaveDeviceWithInvalidTenant() {
        Device device = new Device();
        device.setName("My device");
        device.setType("default");
        device.setTenantId(TenantId.fromUUID((UUID)Uuids.timeBased()));
        org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.saveDevice(device));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testShouldNotPutInCacheRolledbackDeviceProfile() {
        DeviceProfile deviceProfile = this.createDeviceProfile(this.tenantId, "New device Profile" + StringUtils.randomAlphabetic((int)5));
        Device device = new Device();
        device.setType(deviceProfile.getName());
        device.setTenantId(this.tenantId);
        device.setName("My device" + StringUtils.randomAlphabetic((int)5));
        DefaultTransactionDefinition def = new DefaultTransactionDefinition();
        TransactionStatus status = this.platformTransactionManager.getTransaction((TransactionDefinition)def);
        try {
            this.deviceProfileService.saveDeviceProfile(deviceProfile);
            this.deviceService.saveDevice(device);
        }
        finally {
            this.platformTransactionManager.rollback(status);
        }
        DeviceProfile deviceProfileByName = this.deviceProfileService.findDeviceProfileByName(this.tenantId, deviceProfile.getName());
        Assert.assertNull((Object)deviceProfileByName);
    }

    @Test
    public void testAssignDeviceToNonExistentCustomer() {
        Device device = new Device();
        device.setName("My device");
        device.setType("default");
        device.setTenantId(this.tenantId);
        Device savedDevice = this.deviceService.saveDevice(device);
        try {
            org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.assignDeviceToCustomer(this.tenantId, savedDevice.getId(), new CustomerId(Uuids.timeBased())));
        }
        finally {
            this.deviceService.deleteDevice(this.tenantId, savedDevice.getId());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAssignDeviceToCustomerFromDifferentTenant() {
        Device device = new Device();
        device.setName("My device");
        device.setType("default");
        device.setTenantId(this.tenantId);
        Device savedDevice = this.deviceService.saveDevice(device);
        Tenant tenant = new Tenant();
        tenant.setTitle("Test different tenant");
        tenant = this.tenantService.saveTenant(tenant);
        Customer customer = new Customer();
        customer.setTenantId(tenant.getId());
        customer.setTitle("Test different customer");
        Customer savedCustomer = this.customerService.saveCustomer(customer);
        try {
            org.junit.jupiter.api.Assertions.assertThrows(DataValidationException.class, () -> this.deviceService.assignDeviceToCustomer(this.tenantId, savedDevice.getId(), savedCustomer.getId()));
        }
        finally {
            this.deviceService.deleteDevice(this.tenantId, savedDevice.getId());
            this.tenantService.deleteTenant(tenant.getId());
        }
    }

    @Test
    public void testFindDeviceById() {
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("My device");
        device.setType("default");
        Device savedDevice = this.deviceService.saveDevice(device);
        Device foundDevice = this.deviceService.findDeviceById(this.tenantId, savedDevice.getId());
        Assert.assertNotNull((Object)foundDevice);
        Assert.assertEquals((Object)savedDevice, (Object)foundDevice);
        this.deviceService.deleteDevice(this.tenantId, savedDevice.getId());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFindDeviceTypesByTenantId() throws Exception {
        ArrayList<Device> devices = new ArrayList<Device>();
        try {
            Device device2;
            int i;
            for (i = 0; i < 3; ++i) {
                device2 = new Device();
                device2.setTenantId(this.tenantId);
                device2.setName("My device B" + i);
                device2.setType("typeB");
                devices.add(this.deviceService.saveDevice(device2));
            }
            for (i = 0; i < 7; ++i) {
                device2 = new Device();
                device2.setTenantId(this.tenantId);
                device2.setName("My device C" + i);
                device2.setType("typeC");
                devices.add(this.deviceService.saveDevice(device2));
            }
            for (i = 0; i < 9; ++i) {
                device2 = new Device();
                device2.setTenantId(this.tenantId);
                device2.setName("My device A" + i);
                device2.setType("typeA");
                devices.add(this.deviceService.saveDevice(device2));
            }
            List deviceTypes = (List)this.deviceService.findDeviceTypesByTenantId(this.tenantId).get();
            Assert.assertNotNull((Object)deviceTypes);
            Assert.assertEquals((long)3L, (long)deviceTypes.size());
            Assert.assertEquals((Object)"typeA", (Object)((EntitySubtype)deviceTypes.get(0)).getType());
            Assert.assertEquals((Object)"typeB", (Object)((EntitySubtype)deviceTypes.get(1)).getType());
            Assert.assertEquals((Object)"typeC", (Object)((EntitySubtype)deviceTypes.get(2)).getType());
        }
        finally {
            devices.forEach(device -> this.deviceService.deleteDevice(this.tenantId, device.getId()));
        }
    }

    @Test
    public void testDeleteDevice() {
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("My device");
        device.setType("default");
        Device savedDevice = this.deviceService.saveDevice(device);
        Device foundDevice = this.deviceService.findDeviceById(this.tenantId, savedDevice.getId());
        Assert.assertNotNull((Object)foundDevice);
        this.deviceService.deleteDevice(this.tenantId, savedDevice.getId());
        foundDevice = this.deviceService.findDeviceById(this.tenantId, savedDevice.getId());
        Assert.assertNull((Object)foundDevice);
        DeviceCredentials foundDeviceCredentials = this.deviceCredentialsService.findDeviceCredentialsByDeviceId(this.tenantId, savedDevice.getId());
        Assert.assertNull((Object)foundDeviceCredentials);
    }

    @Test
    public void testFindDevicesByTenantId() {
        ArrayList<Device> devices = new ArrayList<Device>();
        for (int i = 0; i < 178; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            device.setName("Device" + i);
            device.setType("default");
            devices.add(this.deviceService.saveDevice(device));
        }
        ArrayList loadedDevices = new ArrayList();
        PageLink pageLink = new PageLink(23);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDevicesByTenantId(this.tenantId, pageLink);
            loadedDevices.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devices, this.idComparator);
        Collections.sort(loadedDevices, this.idComparator);
        Assert.assertEquals(devices, loadedDevices);
        this.deviceService.deleteDevicesByTenantId(this.tenantId);
        pageLink = new PageLink(33);
        pageData = this.deviceService.findDevicesByTenantId(this.tenantId, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertTrue((boolean)pageData.getData().isEmpty());
    }

    @Test
    public void testFindDevicesByTenantIdAndName() {
        String title1 = "Device title 1";
        ArrayList<DeviceInfo> devicesTitle1 = new ArrayList<DeviceInfo>();
        for (int i = 0; i < 143; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title1 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType("default");
            devicesTitle1.add(new DeviceInfo(this.deviceService.saveDevice(device), null, false, "default", false));
        }
        String title2 = "Device title 2";
        ArrayList<DeviceInfo> devicesTitle2 = new ArrayList<DeviceInfo>();
        for (int i = 0; i < 175; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title2 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType("default");
            devicesTitle2.add(new DeviceInfo(this.deviceService.saveDevice(device), null, false, "default", false));
        }
        ArrayList loadedDevicesTitle1 = new ArrayList();
        PageLink pageLink = new PageLink(15, 0, title1);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLink);
            loadedDevicesTitle1.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesTitle1, this.idComparator);
        Collections.sort(loadedDevicesTitle1, this.idComparator);
        Assert.assertEquals(devicesTitle1, loadedDevicesTitle1);
        ArrayList loadedDevicesTitle2 = new ArrayList();
        pageLink = new PageLink(4, 0, title2);
        do {
            pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLink);
            loadedDevicesTitle2.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesTitle2, this.idComparator);
        Collections.sort(loadedDevicesTitle2, this.idComparator);
        Assert.assertEquals(devicesTitle2, loadedDevicesTitle2);
        for (Device device : loadedDevicesTitle1) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4, 0, title1);
        pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        for (Device device : loadedDevicesTitle2) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4, 0, title2);
        pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
    }

    @Test
    public void testFindDevicesByTenantIdAndType() {
        String title1 = "Device title 1";
        String type1 = "typeA";
        ArrayList<Device> devicesType1 = new ArrayList<Device>();
        for (int i = 0; i < 143; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title1 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType(type1);
            devicesType1.add(this.deviceService.saveDevice(device));
        }
        String title2 = "Device title 2";
        String type2 = "typeB";
        ArrayList<Device> devicesType2 = new ArrayList<Device>();
        for (int i = 0; i < 175; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title2 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType(type2);
            devicesType2.add(this.deviceService.saveDevice(device));
        }
        ArrayList loadedDevicesType1 = new ArrayList();
        PageLink pageLink = new PageLink(15);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndType(this.tenantId, type1, pageLink);
            loadedDevicesType1.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesType1, this.idComparator);
        Collections.sort(loadedDevicesType1, this.idComparator);
        Assert.assertEquals(devicesType1, loadedDevicesType1);
        ArrayList loadedDevicesType2 = new ArrayList();
        pageLink = new PageLink(4);
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndType(this.tenantId, type2, pageLink);
            loadedDevicesType2.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesType2, this.idComparator);
        Collections.sort(loadedDevicesType2, this.idComparator);
        Assert.assertEquals(devicesType2, loadedDevicesType2);
        for (Device device : loadedDevicesType1) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4);
        pageData = this.deviceService.findDevicesByTenantIdAndType(this.tenantId, type1, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        for (Device device : loadedDevicesType2) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4);
        pageData = this.deviceService.findDevicesByTenantIdAndType(this.tenantId, type2, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
    }

    @Test
    public void testFindDevicesByTenantIdAndCustomerId() {
        Customer customer = new Customer();
        customer.setTitle("Test customer");
        customer.setTenantId(this.tenantId);
        customer = this.customerService.saveCustomer(customer);
        CustomerId customerId = customer.getId();
        ArrayList<DeviceInfo> devices = new ArrayList<DeviceInfo>();
        for (int i = 0; i < 278; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            device.setName("Device" + i);
            device.setType("default");
            device = this.deviceService.saveDevice(device);
            devices.add(new DeviceInfo(this.deviceService.assignDeviceToCustomer(this.tenantId, device.getId(), customerId), customer.getTitle(), customer.isPublic(), "default", false));
        }
        ArrayList loadedDevices = new ArrayList();
        PageLink pageLink = new PageLink(23);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).customerId(customerId).build(), pageLink);
            loadedDevices.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devices, this.idComparator);
        Collections.sort(loadedDevices, this.idComparator);
        Assert.assertEquals(devices, loadedDevices);
        this.deviceService.unassignCustomerDevices(this.tenantId, customerId);
        pageLink = new PageLink(33);
        pageData = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).customerId(customerId).build(), pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertTrue((boolean)pageData.getData().isEmpty());
    }

    @Test
    public void testFindDevicesByTenantIdCustomerIdAndName() {
        Customer customer = new Customer();
        customer.setTitle("Test customer");
        customer.setTenantId(this.tenantId);
        customer = this.customerService.saveCustomer(customer);
        CustomerId customerId = customer.getId();
        String title1 = "Device title 1";
        ArrayList<Device> devicesTitle1 = new ArrayList<Device>();
        for (int i = 0; i < 175; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title1 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType("default");
            device = this.deviceService.saveDevice(device);
            devicesTitle1.add(this.deviceService.assignDeviceToCustomer(this.tenantId, device.getId(), customerId));
        }
        String title2 = "Device title 2";
        ArrayList<Device> devicesTitle2 = new ArrayList<Device>();
        for (int i = 0; i < 143; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title2 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType("default");
            device = this.deviceService.saveDevice(device);
            devicesTitle2.add(this.deviceService.assignDeviceToCustomer(this.tenantId, device.getId(), customerId));
        }
        ArrayList loadedDevicesTitle1 = new ArrayList();
        PageLink pageLink = new PageLink(15, 0, title1);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndCustomerId(this.tenantId, customerId, pageLink);
            loadedDevicesTitle1.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesTitle1, this.idComparator);
        Collections.sort(loadedDevicesTitle1, this.idComparator);
        Assert.assertEquals(devicesTitle1, loadedDevicesTitle1);
        ArrayList loadedDevicesTitle2 = new ArrayList();
        pageLink = new PageLink(4, 0, title2);
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndCustomerId(this.tenantId, customerId, pageLink);
            loadedDevicesTitle2.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesTitle2, this.idComparator);
        Collections.sort(loadedDevicesTitle2, this.idComparator);
        Assert.assertEquals(devicesTitle2, loadedDevicesTitle2);
        for (Device device : loadedDevicesTitle1) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4, 0, title1);
        pageData = this.deviceService.findDevicesByTenantIdAndCustomerId(this.tenantId, customerId, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        for (Device device : loadedDevicesTitle2) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4, 0, title2);
        pageData = this.deviceService.findDevicesByTenantIdAndCustomerId(this.tenantId, customerId, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        this.customerService.deleteCustomer(this.tenantId, customerId);
    }

    @Test
    public void testFindDevicesByTenantIdCustomerIdAndType() {
        Customer customer = new Customer();
        customer.setTitle("Test customer");
        customer.setTenantId(this.tenantId);
        customer = this.customerService.saveCustomer(customer);
        CustomerId customerId = customer.getId();
        String title1 = "Device title 1";
        String type1 = "typeC";
        ArrayList<Device> devicesType1 = new ArrayList<Device>();
        for (int i = 0; i < 175; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title1 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType(type1);
            device = this.deviceService.saveDevice(device);
            devicesType1.add(this.deviceService.assignDeviceToCustomer(this.tenantId, device.getId(), customerId));
        }
        String title2 = "Device title 2";
        String type2 = "typeD";
        ArrayList<Device> devicesType2 = new ArrayList<Device>();
        for (int i = 0; i < 143; ++i) {
            Device device = new Device();
            device.setTenantId(this.tenantId);
            String suffix = StringUtils.randomAlphanumeric((int)15);
            Object name = title2 + suffix;
            name = i % 2 == 0 ? ((String)name).toLowerCase() : ((String)name).toUpperCase();
            device.setName((String)name);
            device.setType(type2);
            device = this.deviceService.saveDevice(device);
            devicesType2.add(this.deviceService.assignDeviceToCustomer(this.tenantId, device.getId(), customerId));
        }
        ArrayList loadedDevicesType1 = new ArrayList();
        PageLink pageLink = new PageLink(15);
        PageData pageData = null;
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndCustomerIdAndType(this.tenantId, customerId, type1, pageLink);
            loadedDevicesType1.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesType1, this.idComparator);
        Collections.sort(loadedDevicesType1, this.idComparator);
        Assert.assertEquals(devicesType1, loadedDevicesType1);
        ArrayList loadedDevicesType2 = new ArrayList();
        pageLink = new PageLink(4);
        do {
            pageData = this.deviceService.findDevicesByTenantIdAndCustomerIdAndType(this.tenantId, customerId, type2, pageLink);
            loadedDevicesType2.addAll(pageData.getData());
            if (!pageData.hasNext()) continue;
            pageLink = pageLink.nextPageLink();
        } while (pageData.hasNext());
        Collections.sort(devicesType2, this.idComparator);
        Collections.sort(loadedDevicesType2, this.idComparator);
        Assert.assertEquals(devicesType2, loadedDevicesType2);
        for (Device device : loadedDevicesType1) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4);
        pageData = this.deviceService.findDevicesByTenantIdAndCustomerIdAndType(this.tenantId, customerId, type1, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        for (Device device : loadedDevicesType2) {
            this.deviceService.deleteDevice(this.tenantId, device.getId());
        }
        pageLink = new PageLink(4);
        pageData = this.deviceService.findDevicesByTenantIdAndCustomerIdAndType(this.tenantId, customerId, type2, pageLink);
        Assert.assertFalse((boolean)pageData.hasNext());
        Assert.assertEquals((long)0L, (long)pageData.getData().size());
        this.customerService.deleteCustomer(this.tenantId, customerId);
    }

    @Test
    public void testCleanCacheIfDeviceRenamed() {
        String deviceNameBeforeRename = StringUtils.randomAlphanumeric((int)15);
        String deviceNameAfterRename = StringUtils.randomAlphanumeric((int)15);
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName(deviceNameBeforeRename);
        device.setType("default");
        this.deviceService.saveDevice(device);
        Device savedDevice = this.deviceService.findDeviceByTenantIdAndName(this.tenantId, deviceNameBeforeRename);
        savedDevice.setName(deviceNameAfterRename);
        this.deviceService.saveDevice(savedDevice);
        Device renamedDevice = this.deviceService.findDeviceByTenantIdAndName(this.tenantId, deviceNameBeforeRename);
        Assert.assertNull((String)"Can't find device by name in cache if it was renamed", (Object)renamedDevice);
        this.deviceService.deleteDevice(this.tenantId, savedDevice.getId());
    }

    @Test
    public void testFindDeviceInfoByTenantId() {
        Customer customer = new Customer();
        customer.setTitle("Customer X");
        customer.setTenantId(this.tenantId);
        Customer savedCustomer = this.customerService.saveCustomer(customer);
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("default");
        device.setType("default");
        device.setLabel("label");
        device.setCustomerId(savedCustomer.getId());
        Device savedDevice = this.deviceService.saveDevice(device);
        PageLink pageLinkWithLabel = new PageLink(100, 0, "label");
        List deviceInfosWithLabel = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLinkWithLabel).getData();
        Assert.assertFalse((boolean)deviceInfosWithLabel.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithLabel.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getLabel().equals(savedDevice.getLabel())));
        PageLink pageLinkWithCustomer = new PageLink(100, 0, savedCustomer.getTitle());
        List deviceInfosWithCustomer = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLinkWithCustomer).getData();
        Assert.assertFalse((boolean)deviceInfosWithCustomer.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithCustomer.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getCustomerId().equals((Object)savedCustomer.getId()) && d.getCustomerTitle().equals(savedCustomer.getTitle())));
        PageLink pageLinkWithType = new PageLink(100, 0, device.getType());
        List deviceInfosWithType = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).build(), pageLinkWithType).getData();
        Assert.assertFalse((boolean)deviceInfosWithType.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithType.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getType().equals(device.getType())));
    }

    @Test
    public void testFindDeviceInfoByTenantIdAndType() {
        Customer customer = new Customer();
        customer.setTitle("Customer X");
        customer.setTenantId(this.tenantId);
        Customer savedCustomer = this.customerService.saveCustomer(customer);
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("default");
        device.setType("default");
        device.setLabel("label");
        device.setCustomerId(savedCustomer.getId());
        Device savedDevice = this.deviceService.saveDevice(device);
        PageLink pageLinkWithLabel = new PageLink(100, 0, "label");
        List deviceInfosWithLabel = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).type(device.getType()).build(), pageLinkWithLabel).getData();
        Assert.assertFalse((boolean)deviceInfosWithLabel.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithLabel.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getDeviceProfileName().equals(savedDevice.getType()) && d.getLabel().equals(savedDevice.getLabel())));
        PageLink pageLinkWithCustomer = new PageLink(100, 0, savedCustomer.getTitle());
        List deviceInfosWithCustomer = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).type(device.getType()).build(), pageLinkWithCustomer).getData();
        Assert.assertFalse((boolean)deviceInfosWithCustomer.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithCustomer.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getDeviceProfileName().equals(savedDevice.getType()) && d.getCustomerId().equals((Object)savedCustomer.getId()) && d.getCustomerTitle().equals(savedCustomer.getTitle())));
    }

    @Test
    public void testFindDeviceInfoByTenantIdAndDeviceProfileId() {
        Customer customer = new Customer();
        customer.setTitle("Customer X");
        customer.setTenantId(this.tenantId);
        Customer savedCustomer = this.customerService.saveCustomer(customer);
        Device device = new Device();
        device.setTenantId(this.tenantId);
        device.setName("default");
        device.setLabel("label");
        device.setCustomerId(savedCustomer.getId());
        Device savedDevice = this.deviceService.saveDevice(device);
        PageLink pageLinkWithLabel = new PageLink(100, 0, "label");
        List deviceInfosWithLabel = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).deviceProfileId(savedDevice.getDeviceProfileId()).build(), pageLinkWithLabel).getData();
        Assert.assertFalse((boolean)deviceInfosWithLabel.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithLabel.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getDeviceProfileId().equals((Object)savedDevice.getDeviceProfileId()) && d.getLabel().equals(savedDevice.getLabel())));
        PageLink pageLinkWithCustomer = new PageLink(100, 0, savedCustomer.getTitle());
        List deviceInfosWithCustomer = this.deviceService.findDeviceInfosByFilter(DeviceInfoFilter.builder().tenantId(this.tenantId).deviceProfileId(savedDevice.getDeviceProfileId()).build(), pageLinkWithCustomer).getData();
        Assert.assertFalse((boolean)deviceInfosWithCustomer.isEmpty());
        Assert.assertTrue((boolean)deviceInfosWithCustomer.stream().anyMatch(d -> d.getId().equals((Object)savedDevice.getId()) && d.getTenantId().equals((Object)this.tenantId) && d.getDeviceProfileId().equals((Object)savedDevice.getDeviceProfileId()) && d.getCustomerId().equals((Object)savedCustomer.getId()) && d.getCustomerTitle().equals(savedCustomer.getTitle())));
    }

    @Test
    public void testDeleteDeviceIfReferencedInCalculatedField() {
        Device device = this.saveDevice(this.tenantId, "Test Device");
        Device deviceWithCf = this.saveDevice(this.tenantId, "Device with CF");
        CalculatedField calculatedField = new CalculatedField();
        calculatedField.setTenantId(this.tenantId);
        calculatedField.setName("Test CF");
        calculatedField.setType(CalculatedFieldType.SIMPLE);
        calculatedField.setEntityId((EntityId)deviceWithCf.getId());
        SimpleCalculatedFieldConfiguration config = new SimpleCalculatedFieldConfiguration();
        Argument argument = new Argument();
        argument.setRefEntityId((EntityId)device.getId());
        ReferencedEntityKey refEntityKey = new ReferencedEntityKey("temperature", ArgumentType.TS_LATEST, null);
        argument.setRefEntityKey(refEntityKey);
        config.setArguments(Map.of("T", argument));
        config.setExpression("T - (100 - H) / 5");
        Output output = new Output();
        output.setName("output");
        output.setType(OutputType.TIME_SERIES);
        config.setOutput(output);
        calculatedField.setConfiguration((CalculatedFieldConfiguration)config);
        CalculatedField savedCalculatedField = this.calculatedFieldService.save(calculatedField);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.deviceService.deleteDevice(this.tenantId, device.getId())).isInstanceOf(DataValidationException.class)).hasMessage("Can't delete device that has entity views or is referenced in calculated fields!");
        this.calculatedFieldService.deleteCalculatedField(this.tenantId, savedCalculatedField.getId());
    }
}

