/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.server.service.sync.vc;

import jakarta.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.FalseFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.NameFileFilter;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Service;
import org.thingsboard.server.common.data.EntityType;
import org.thingsboard.server.common.data.StringUtils;
import org.thingsboard.server.common.data.id.EntityId;
import org.thingsboard.server.common.data.id.EntityIdFactory;
import org.thingsboard.server.common.data.id.TenantId;
import org.thingsboard.server.common.data.page.PageData;
import org.thingsboard.server.common.data.page.PageLink;
import org.thingsboard.server.common.data.sync.vc.BranchInfo;
import org.thingsboard.server.common.data.sync.vc.EntityVersion;
import org.thingsboard.server.common.data.sync.vc.RepositorySettings;
import org.thingsboard.server.common.data.sync.vc.VersionCreationResult;
import org.thingsboard.server.common.data.sync.vc.VersionedEntityInfo;
import org.thingsboard.server.service.sync.vc.GitRepository;
import org.thingsboard.server.service.sync.vc.GitRepositoryService;
import org.thingsboard.server.service.sync.vc.PendingCommit;

@Service
public class DefaultGitRepositoryService
implements GitRepositoryService {
    private static final Logger log = LoggerFactory.getLogger(DefaultGitRepositoryService.class);
    @Value(value="${java.io.tmpdir}/repositories")
    private String defaultFolder;
    @Value(value="${vc.git.repositories-folder:${java.io.tmpdir}/repositories}")
    private String repositoriesFolder;
    private final Map<TenantId, GitRepository> repositories = new ConcurrentHashMap<TenantId, GitRepository>();

    @PostConstruct
    public void init() {
        if (StringUtils.isEmpty((String)this.repositoriesFolder)) {
            this.repositoriesFolder = this.defaultFolder;
        }
    }

    @Override
    public Set<TenantId> getActiveRepositoryTenants() {
        HashSet<TenantId> tenants = new HashSet<TenantId>(this.repositories.keySet());
        tenants.remove(TenantId.SYS_TENANT_ID);
        return tenants;
    }

    @Override
    public void prepareCommit(PendingCommit commit) {
        GitRepository repository = this.checkRepository(commit.getTenantId());
        String branch = commit.getBranch();
        try {
            List<String> branches = repository.listBranches().stream().map(BranchInfo::getName).toList();
            if (repository.getSettings().isLocalOnly()) {
                if (branches.contains(commit.getBranch())) {
                    repository.checkoutBranch(commit.getBranch());
                } else {
                    repository.createAndCheckoutOrphanBranch(commit.getBranch());
                }
                repository.resetAndClean();
            } else {
                repository.createAndCheckoutOrphanBranch(commit.getWorkingBranch());
                repository.resetAndClean();
                if (branches.contains(branch)) {
                    repository.merge(branch);
                }
            }
        }
        catch (IOException | GitAPIException gitAPIException) {
            throw new RuntimeException(gitAPIException);
        }
    }

    @Override
    public void deleteFolderContent(PendingCommit commit, String folder, boolean recursively) throws IOException {
        GitRepository repository = this.checkRepository(commit.getTenantId());
        Path workDir = Path.of(repository.getDirectory(), new String[0]);
        if (recursively) {
            Collection dirs = FileUtils.listFilesAndDirs((File)workDir.toFile(), (IOFileFilter)FalseFileFilter.FALSE, (IOFileFilter)new NameFileFilter(".git").negate());
            for (File dir : dirs) {
                if (!dir.getName().equals(folder)) continue;
                FileUtils.deleteDirectory((File)dir);
            }
        } else {
            FileUtils.deleteDirectory((File)Path.of(repository.getDirectory(), folder).toFile());
        }
    }

    @Override
    public void add(PendingCommit commit, String relativePath, String entityDataJson) throws IOException {
        GitRepository repository = this.checkRepository(commit.getTenantId());
        FileUtils.write((File)Path.of(repository.getDirectory(), relativePath).toFile(), (CharSequence)entityDataJson, (Charset)StandardCharsets.UTF_8);
    }

    @Override
    public VersionCreationResult push(PendingCommit commit) {
        GitRepository repository = this.checkRepository(commit.getTenantId());
        try {
            repository.add(".");
            VersionCreationResult result = new VersionCreationResult();
            GitRepository.Status status = repository.status();
            result.setAdded(status.getAdded().size());
            result.setModified(status.getModified().size());
            result.setRemoved(status.getRemoved().size());
            if (result.getAdded() > 0 || result.getModified() > 0 || result.getRemoved() > 0) {
                GitRepository.Commit gitCommit = repository.commit(commit.getVersionName(), commit.getAuthorName(), commit.getAuthorEmail());
                repository.push(commit.getWorkingBranch(), commit.getBranch());
                result.setVersion(this.toVersion(gitCommit));
            }
            VersionCreationResult versionCreationResult = result;
            return versionCreationResult;
        }
        catch (GitAPIException gitAPIException) {
            throw new RuntimeException(gitAPIException);
        }
        finally {
            this.cleanUp(commit);
        }
    }

    @Override
    public void cleanUp(PendingCommit commit) {
        GitRepository repository;
        block4: {
            log.debug("[{}] Cleanup tenant repository started.", (Object)commit.getTenantId());
            repository = this.checkRepository(commit.getTenantId());
            try {
                repository.createAndCheckoutOrphanBranch(EntityId.NULL_UUID.toString());
            }
            catch (Exception e) {
                if (e.getMessage().contains("NO_CHANGE")) break block4;
                throw e;
            }
        }
        repository.resetAndClean();
        repository.deleteLocalBranchIfExists(commit.getWorkingBranch());
        log.debug("[{}] Cleanup tenant repository completed.", (Object)commit.getTenantId());
    }

    @Override
    public void abort(PendingCommit commit) {
        this.cleanUp(commit);
    }

    @Override
    public void fetch(TenantId tenantId) throws GitAPIException {
        GitRepository repository = this.checkRepository(tenantId);
        log.debug("[{}] Fetching tenant repository.", (Object)tenantId);
        repository.fetch();
        log.debug("[{}] Fetched tenant repository.", (Object)tenantId);
    }

    @Override
    public String getFileContentAtCommit(TenantId tenantId, String relativePath, String versionId) {
        GitRepository repository = this.checkRepository(tenantId);
        return new String(repository.getFileContentAtCommit(relativePath, versionId), StandardCharsets.UTF_8);
    }

    @Override
    public List<GitRepository.Diff> getVersionsDiffList(TenantId tenantId, String path, String versionId1, String versionId2) throws IOException {
        GitRepository repository = this.checkRepository(tenantId);
        return repository.getDiffList(versionId1, versionId2, path);
    }

    @Override
    public String getContentsDiff(TenantId tenantId, String content1, String content2) throws IOException {
        GitRepository repository = this.checkRepository(tenantId);
        return repository.getContentsDiff(content1, content2);
    }

    @Override
    public List<BranchInfo> listBranches(TenantId tenantId) {
        GitRepository repository = this.checkRepository(tenantId);
        try {
            return repository.listBranches();
        }
        catch (GitAPIException gitAPIException) {
            throw new RuntimeException(gitAPIException);
        }
    }

    private GitRepository checkRepository(TenantId tenantId) {
        GitRepository gitRepository = Optional.ofNullable(this.repositories.get(tenantId)).orElseThrow(() -> new IllegalStateException("Repository is not initialized"));
        if (!GitRepository.exists(gitRepository.getDirectory())) {
            try {
                return this.openOrCloneRepository(tenantId, gitRepository.getSettings(), false);
            }
            catch (Exception e) {
                throw new IllegalStateException("Could not initialize the repository: " + e.getMessage(), e);
            }
        }
        return gitRepository;
    }

    @Override
    public PageData<EntityVersion> listVersions(TenantId tenantId, String branch, String path, PageLink pageLink) throws Exception {
        GitRepository repository = this.checkRepository(tenantId);
        return repository.listCommits(branch, path, pageLink).mapData(this::toVersion);
    }

    public Pattern buildPattern(EntityType entityType, boolean group) {
        Object prefix = ".*";
        if (group) {
            prefix = (String)prefix + "groups\\/";
        }
        prefix = (String)prefix + entityType.name().toLowerCase() + "\\/";
        return Pattern.compile((String)prefix + "[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}.json");
    }

    @Override
    public Stream<VersionedEntityInfo> listEntitiesAtVersion(TenantId tenantId, String versionId, String folder, EntityType entityType, boolean isGroup, boolean recursive) throws Exception {
        GitRepository repository = this.checkRepository(tenantId);
        if (entityType != null) {
            String path = recursive ? folder : StringUtils.emptyIfNull((String)folder) + entityType.name().toLowerCase();
            Pattern typePattern = this.buildPattern(entityType, false);
            Pattern groupPattern = this.buildPattern(entityType, true);
            return repository.listAllFilesAtCommit(versionId, path).stream().filter(filePath -> typePattern.matcher((CharSequence)filePath).matches()).filter(filePath -> groupPattern.matcher((CharSequence)filePath).matches() == isGroup).map(filePath -> {
                String[] parts = filePath.split("/");
                String uuidStr = parts[parts.length - 1];
                EntityId entityId = EntityIdFactory.getByTypeAndUuid((EntityType)entityType, (String)uuidStr.substring(0, 36));
                return new VersionedEntityInfo(entityId, filePath);
            }).sorted(Comparator.comparing(VersionedEntityInfo::getPath, Comparator.comparingInt(String::length)).thenComparing(VersionedEntityInfo::getPath, String::compareTo));
        }
        HashMap<EntityType, Pattern> typePatterns = new HashMap<EntityType, Pattern>();
        HashMap<EntityType, Pattern> groupPatterns = new HashMap<EntityType, Pattern>();
        for (EntityType et : EntityType.values()) {
            groupPatterns.put(et, this.buildPattern(et, true));
            typePatterns.put(et, this.buildPattern(et, false));
        }
        return repository.listAllFilesAtCommit(versionId, folder).stream().map(filePath -> {
            for (Map.Entry pair : groupPatterns.entrySet()) {
                if (!((Pattern)pair.getValue()).matcher((CharSequence)filePath).matches()) continue;
                return Pair.of((Object)EntityType.ENTITY_GROUP, (Object)filePath);
            }
            for (Map.Entry pair : typePatterns.entrySet()) {
                if (!((Pattern)pair.getValue()).matcher((CharSequence)filePath).matches()) continue;
                return Pair.of((Object)((EntityType)pair.getKey()), (Object)filePath);
            }
            return null;
        }).filter(Objects::nonNull).map(pair -> {
            String[] parts = ((String)pair.getSecond()).split("/");
            String uuidStr = parts[parts.length - 1];
            EntityId entityId = EntityIdFactory.getByTypeAndUuid((EntityType)((EntityType)pair.getFirst()), (String)uuidStr.substring(0, 36));
            return new VersionedEntityInfo(entityId, (String)pair.getSecond());
        }).sorted(Comparator.comparing(VersionedEntityInfo::getPath, Comparator.comparingInt(String::length)).thenComparing(VersionedEntityInfo::getPath, String::compareTo));
    }

    @Override
    public List<GitRepository.RepoFile> listFiles(TenantId tenantId, String versionId, String path, int depth) {
        GitRepository repository = this.checkRepository(tenantId);
        return repository.listFilesAtCommit(versionId, path, depth);
    }

    @Override
    public void testRepository(TenantId tenantId, RepositorySettings settings) throws Exception {
        Path testDirectory = Path.of(this.repositoriesFolder, "repo-test-" + UUID.randomUUID());
        GitRepository.test(settings, testDirectory.toFile());
    }

    @Override
    public void initRepository(TenantId tenantId, RepositorySettings settings, boolean fetch) throws Exception {
        if (!settings.isLocalOnly()) {
            this.clearRepository(tenantId);
        }
        this.openOrCloneRepository(tenantId, settings, fetch);
    }

    @Override
    public RepositorySettings getRepositorySettings(TenantId tenantId) throws Exception {
        GitRepository gitRepository = this.repositories.get(tenantId);
        return gitRepository != null ? gitRepository.getSettings() : null;
    }

    @Override
    public void clearRepository(TenantId tenantId) throws IOException {
        GitRepository repository = this.repositories.remove(tenantId);
        if (repository != null) {
            log.debug("[{}] Clear tenant repository started.", (Object)tenantId);
            FileUtils.deleteDirectory((File)new File(repository.getDirectory()));
            log.debug("[{}] Clear tenant repository completed.", (Object)tenantId);
        }
    }

    private EntityVersion toVersion(GitRepository.Commit commit) {
        return new EntityVersion(commit.getTimestamp(), commit.getId(), commit.getMessage(), this.getAuthor(commit));
    }

    private String getAuthor(GitRepository.Commit commit) {
        String author = String.format("<%s>", commit.getAuthorEmail());
        if (StringUtils.isNotBlank((String)commit.getAuthorName())) {
            author = String.format("%s %s", commit.getAuthorName(), author);
        }
        return author;
    }

    public static EntityId fromRelativePath(String path) {
        EntityType entityType = EntityType.valueOf((String)StringUtils.substringBefore((String)path, (String)"/").toUpperCase());
        String entityId = StringUtils.substringBetween((String)path, (String)"/", (String)".json");
        return EntityIdFactory.getByTypeAndUuid((EntityType)entityType, (String)entityId);
    }

    private GitRepository openOrCloneRepository(TenantId tenantId, RepositorySettings settings, boolean fetch) throws Exception {
        log.debug("[{}] Init tenant repository started.", (Object)tenantId);
        Path repositoryDirectory = Path.of(this.repositoriesFolder, new String[]{settings.isLocalOnly() ? "local_" + settings.getRepositoryUri() : tenantId.getId().toString()});
        GitRepository repository = GitRepository.openOrClone(repositoryDirectory, settings, fetch);
        this.repositories.put(tenantId, repository);
        log.debug("[{}] Init tenant repository completed.", (Object)tenantId);
        return repository;
    }
}

