/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.server.transport.snmp;

import java.beans.ConstructorProperties;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;
import org.thingsboard.server.common.data.Device;
import org.thingsboard.server.common.data.DeviceProfile;
import org.thingsboard.server.common.data.DeviceTransportType;
import org.thingsboard.server.common.data.device.data.DeviceTransportConfiguration;
import org.thingsboard.server.common.data.device.data.SnmpDeviceTransportConfiguration;
import org.thingsboard.server.common.data.device.profile.SnmpDeviceProfileTransportConfiguration;
import org.thingsboard.server.common.data.id.DeviceId;
import org.thingsboard.server.common.data.id.DeviceProfileId;
import org.thingsboard.server.common.data.security.DeviceCredentials;
import org.thingsboard.server.common.data.security.DeviceCredentialsType;
import org.thingsboard.server.common.transport.DeviceUpdatedEvent;
import org.thingsboard.server.common.transport.SessionMsgListener;
import org.thingsboard.server.common.transport.TransportContext;
import org.thingsboard.server.common.transport.TransportDeviceProfileCache;
import org.thingsboard.server.common.transport.TransportService;
import org.thingsboard.server.common.transport.TransportServiceCallback;
import org.thingsboard.server.common.transport.auth.SessionInfoCreator;
import org.thingsboard.server.common.transport.auth.ValidateDeviceCredentialsResponse;
import org.thingsboard.server.gen.transport.TransportProtos;
import org.thingsboard.server.queue.util.AfterStartUp;
import org.thingsboard.server.queue.util.TbSnmpTransportComponent;
import org.thingsboard.server.transport.snmp.service.ProtoTransportEntityService;
import org.thingsboard.server.transport.snmp.service.SnmpAuthService;
import org.thingsboard.server.transport.snmp.service.SnmpTransportBalancingService;
import org.thingsboard.server.transport.snmp.service.SnmpTransportService;
import org.thingsboard.server.transport.snmp.session.DeviceSessionContext;

@TbSnmpTransportComponent
@Component
public class SnmpTransportContext
extends TransportContext {
    private static final Logger log = LoggerFactory.getLogger(SnmpTransportContext.class);
    private final SnmpTransportService snmpTransportService;
    private final TransportDeviceProfileCache deviceProfileCache;
    private final TransportService transportService;
    private final ProtoTransportEntityService protoEntityService;
    private final SnmpTransportBalancingService balancingService;
    private final SnmpAuthService snmpAuthService;
    private final Map<DeviceId, DeviceSessionContext> sessions = new ConcurrentHashMap<DeviceId, DeviceSessionContext>();
    private final Collection<DeviceId> allSnmpDevicesIds = new ConcurrentLinkedDeque<DeviceId>();

    @AfterStartUp(order=2147482648)
    public void fetchDevicesAndEstablishSessions() {
        log.info("Initializing SNMP devices sessions");
        int batchIndex = 0;
        int batchSize = 512;
        boolean nextBatchExists = true;
        while (nextBatchExists) {
            TransportProtos.GetSnmpDevicesResponseMsg snmpDevicesResponse = this.protoEntityService.getSnmpDevicesIds(batchIndex, batchSize);
            snmpDevicesResponse.getIdsList().stream().map(id -> new DeviceId(UUID.fromString(id))).peek(this.allSnmpDevicesIds::add).filter(deviceId -> this.balancingService.isManagedByCurrentTransport(deviceId.getId())).map(this.protoEntityService::getDeviceById).forEach(device -> this.getExecutor().execute(() -> this.establishDeviceSession((Device)device)));
            nextBatchExists = snmpDevicesResponse.getHasNextPage();
            ++batchIndex;
        }
        log.debug("Found all SNMP devices ids: {}", this.allSnmpDevicesIds);
    }

    private void establishDeviceSession(Device device) {
        DeviceSessionContext deviceSessionContext;
        if (device == null) {
            return;
        }
        log.info("Establishing SNMP session for device {}", (Object)device.getId());
        DeviceProfileId deviceProfileId = device.getDeviceProfileId();
        DeviceProfile deviceProfile = this.deviceProfileCache.get(deviceProfileId);
        DeviceCredentials credentials = this.protoEntityService.getDeviceCredentialsByDeviceId(device.getId());
        if (credentials.getCredentialsType() != DeviceCredentialsType.ACCESS_TOKEN) {
            log.warn("[{}] Expected credentials type is {} but found {}", new Object[]{device.getId(), DeviceCredentialsType.ACCESS_TOKEN, credentials.getCredentialsType()});
            return;
        }
        SnmpDeviceProfileTransportConfiguration profileTransportConfiguration = (SnmpDeviceProfileTransportConfiguration)deviceProfile.getProfileData().getTransportConfiguration();
        SnmpDeviceTransportConfiguration deviceTransportConfiguration = (SnmpDeviceTransportConfiguration)device.getDeviceData().getTransportConfiguration();
        try {
            deviceSessionContext = new DeviceSessionContext(device, deviceProfile, credentials.getCredentialsId(), profileTransportConfiguration, deviceTransportConfiguration, this);
            this.registerSessionMsgListener(deviceSessionContext);
        }
        catch (Exception e) {
            log.error("Failed to establish session for SNMP device {}: {}", (Object)device.getId(), (Object)e.toString());
            return;
        }
        this.sessions.put(device.getId(), deviceSessionContext);
        this.snmpTransportService.createQueryingTasks(deviceSessionContext);
        log.info("Established SNMP device session for device {}", (Object)device.getId());
    }

    private void updateDeviceSession(DeviceSessionContext sessionContext, Device device, DeviceProfile deviceProfile) {
        log.info("Updating SNMP session for device {}", (Object)device.getId());
        DeviceCredentials credentials = this.protoEntityService.getDeviceCredentialsByDeviceId(device.getId());
        if (credentials.getCredentialsType() != DeviceCredentialsType.ACCESS_TOKEN) {
            log.warn("[{}] Expected credentials type is {} but found {}", new Object[]{device.getId(), DeviceCredentialsType.ACCESS_TOKEN, credentials.getCredentialsType()});
            this.destroyDeviceSession(sessionContext);
            return;
        }
        SnmpDeviceProfileTransportConfiguration newProfileTransportConfiguration = (SnmpDeviceProfileTransportConfiguration)deviceProfile.getProfileData().getTransportConfiguration();
        SnmpDeviceTransportConfiguration newDeviceTransportConfiguration = (SnmpDeviceTransportConfiguration)device.getDeviceData().getTransportConfiguration();
        try {
            if (!newProfileTransportConfiguration.equals((Object)sessionContext.getProfileTransportConfiguration())) {
                sessionContext.setProfileTransportConfiguration(newProfileTransportConfiguration);
                sessionContext.initializeTarget(newProfileTransportConfiguration, newDeviceTransportConfiguration);
                this.snmpTransportService.cancelQueryingTasks(sessionContext);
                this.snmpTransportService.createQueryingTasks(sessionContext);
            } else if (!newDeviceTransportConfiguration.equals((Object)sessionContext.getDeviceTransportConfiguration())) {
                sessionContext.setDeviceTransportConfiguration(newDeviceTransportConfiguration);
                sessionContext.initializeTarget(newProfileTransportConfiguration, newDeviceTransportConfiguration);
            } else {
                log.trace("Configuration of the device {} was not updated", (Object)device);
            }
        }
        catch (Exception e) {
            log.error("Failed to update session for SNMP device {}: {}", (Object)sessionContext.getDeviceId(), (Object)e.getMessage());
            this.destroyDeviceSession(sessionContext);
        }
    }

    private void destroyDeviceSession(DeviceSessionContext sessionContext) {
        if (sessionContext == null) {
            return;
        }
        log.info("Destroying SNMP device session for device {}", (Object)sessionContext.getDevice().getId());
        sessionContext.close();
        this.snmpAuthService.cleanUpSnmpAuthInfo(sessionContext);
        this.transportService.deregisterSession(sessionContext.getSessionInfo());
        this.snmpTransportService.cancelQueryingTasks(sessionContext);
        this.sessions.remove(sessionContext.getDeviceId());
        log.trace("Unregistered and removed session");
    }

    private void registerSessionMsgListener(final DeviceSessionContext deviceSessionContext) {
        this.transportService.process(DeviceTransportType.SNMP, TransportProtos.ValidateDeviceTokenRequestMsg.newBuilder().setToken(deviceSessionContext.getToken()).build(), (TransportServiceCallback)new TransportServiceCallback<ValidateDeviceCredentialsResponse>(){

            public void onSuccess(ValidateDeviceCredentialsResponse msg) {
                if (msg.hasDeviceInfo()) {
                    TransportProtos.SessionInfoProto sessionInfo = SessionInfoCreator.create((ValidateDeviceCredentialsResponse)msg, (TransportContext)SnmpTransportContext.this, (UUID)UUID.randomUUID());
                    SnmpTransportContext.this.transportService.registerAsyncSession(sessionInfo, (SessionMsgListener)deviceSessionContext);
                    SnmpTransportContext.this.transportService.process(sessionInfo, TransportProtos.SubscribeToAttributeUpdatesMsg.newBuilder().build(), TransportServiceCallback.EMPTY);
                    SnmpTransportContext.this.transportService.process(sessionInfo, TransportProtos.SubscribeToRPCMsg.newBuilder().build(), TransportServiceCallback.EMPTY);
                    deviceSessionContext.setSessionInfo(sessionInfo);
                    deviceSessionContext.setDeviceInfo(msg.getDeviceInfo());
                    deviceSessionContext.setConnected(true);
                } else {
                    log.warn("[{}] Failed to process device auth", (Object)deviceSessionContext.getDeviceId());
                }
            }

            public void onError(Throwable e) {
                log.warn("[{}] Failed to process device auth: {}", (Object)deviceSessionContext.getDeviceId(), (Object)e);
            }
        });
    }

    @EventListener(value={DeviceUpdatedEvent.class})
    public void onDeviceUpdatedOrCreated(DeviceUpdatedEvent deviceUpdatedEvent) {
        Device device = deviceUpdatedEvent.getDevice();
        log.trace("Got creating or updating device event for device {}", (Object)device);
        DeviceTransportType transportType = Optional.ofNullable(device.getDeviceData().getTransportConfiguration()).map(DeviceTransportConfiguration::getType).orElse(null);
        if (!this.allSnmpDevicesIds.contains(device.getId())) {
            if (transportType != DeviceTransportType.SNMP) {
                return;
            }
            this.allSnmpDevicesIds.add(device.getId());
            if (this.balancingService.isManagedByCurrentTransport(device.getId().getId())) {
                this.establishDeviceSession(device);
            }
        } else if (this.balancingService.isManagedByCurrentTransport(device.getId().getId())) {
            DeviceSessionContext sessionContext = this.sessions.get(device.getId());
            if (transportType == DeviceTransportType.SNMP) {
                if (sessionContext != null) {
                    this.updateDeviceSession(sessionContext, device, this.deviceProfileCache.get(device.getDeviceProfileId()));
                } else {
                    this.establishDeviceSession(device);
                }
            } else {
                log.trace("Transport type was changed to {}", (Object)transportType);
                this.destroyDeviceSession(sessionContext);
            }
        }
    }

    public void onDeviceDeleted(DeviceSessionContext sessionContext) {
        this.destroyDeviceSession(sessionContext);
    }

    public void onDeviceProfileUpdated(DeviceProfile deviceProfile, DeviceSessionContext sessionContext) {
        this.updateDeviceSession(sessionContext, sessionContext.getDevice(), deviceProfile);
    }

    public void onSnmpTransportListChanged() {
        log.trace("SNMP transport list changed. Updating sessions");
        LinkedList<DeviceId> deleted = new LinkedList<DeviceId>();
        for (DeviceId deviceId : this.allSnmpDevicesIds) {
            if (this.balancingService.isManagedByCurrentTransport(deviceId.getId())) {
                if (this.sessions.containsKey(deviceId)) continue;
                Device device = this.protoEntityService.getDeviceById(deviceId);
                if (device != null) {
                    log.info("SNMP device {} is now managed by current transport node", (Object)deviceId);
                    this.establishDeviceSession(device);
                    continue;
                }
                deleted.add(deviceId);
                continue;
            }
            Optional.ofNullable(this.sessions.get(deviceId)).ifPresent(sessionContext -> {
                log.info("SNMP session for device {} is not managed by current transport node anymore", (Object)deviceId);
                this.destroyDeviceSession((DeviceSessionContext)((Object)sessionContext));
            });
        }
        log.trace("Removing deleted SNMP devices: {}", deleted);
        this.allSnmpDevicesIds.removeAll(deleted);
    }

    public Collection<DeviceSessionContext> getSessions() {
        return this.sessions.values();
    }

    @ConstructorProperties(value={"snmpTransportService", "deviceProfileCache", "transportService", "protoEntityService", "balancingService", "snmpAuthService"})
    public SnmpTransportContext(SnmpTransportService snmpTransportService, TransportDeviceProfileCache deviceProfileCache, TransportService transportService, ProtoTransportEntityService protoEntityService, SnmpTransportBalancingService balancingService, SnmpAuthService snmpAuthService) {
        this.snmpTransportService = snmpTransportService;
        this.deviceProfileCache = deviceProfileCache;
        this.transportService = transportService;
        this.protoEntityService = protoEntityService;
        this.balancingService = balancingService;
        this.snmpAuthService = snmpAuthService;
    }

    public SnmpTransportService getSnmpTransportService() {
        return this.snmpTransportService;
    }

    public SnmpAuthService getSnmpAuthService() {
        return this.snmpAuthService;
    }
}

