/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.script.api;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.thingsboard.common.util.JacksonUtil;
import org.thingsboard.common.util.ThingsBoardExecutors;
import org.thingsboard.script.api.BlockedScriptInfo;
import org.thingsboard.script.api.ScriptInvokeService;
import org.thingsboard.script.api.ScriptStatCallback;
import org.thingsboard.script.api.ScriptType;
import org.thingsboard.script.api.TbScriptException;
import org.thingsboard.script.api.TbScriptExecutionTask;
import org.thingsboard.server.common.data.id.CustomerId;
import org.thingsboard.server.common.data.id.TenantId;

public abstract class AbstractScriptInvokeService
implements ScriptInvokeService {
    private static final Logger log = LoggerFactory.getLogger(AbstractScriptInvokeService.class);
    protected final Map<UUID, BlockedScriptInfo> disabledScripts = new ConcurrentHashMap<UUID, BlockedScriptInfo>();
    private final AtomicInteger pushedMsgs = new AtomicInteger(0);
    private final AtomicInteger invokeMsgs = new AtomicInteger(0);
    private final AtomicInteger evalMsgs = new AtomicInteger(0);
    protected final AtomicInteger failedMsgs = new AtomicInteger(0);
    protected final AtomicInteger timeoutMsgs = new AtomicInteger(0);
    private final FutureCallback<UUID> evalCallback = new ScriptStatCallback<UUID>(this.evalMsgs, this.timeoutMsgs, this.failedMsgs);
    private final FutureCallback<Object> invokeCallback = new ScriptStatCallback<Object>(this.invokeMsgs, this.timeoutMsgs, this.failedMsgs);
    protected ScheduledExecutorService timeoutExecutorService;

    protected long getMaxEvalRequestsTimeout() {
        return this.getMaxInvokeRequestsTimeout();
    }

    protected abstract long getMaxInvokeRequestsTimeout();

    protected abstract long getMaxScriptBodySize();

    protected abstract long getMaxTotalArgsSize();

    protected abstract long getMaxResultSize();

    protected abstract int getMaxBlackListDurationSec();

    protected abstract int getMaxErrors();

    protected abstract boolean isStatsEnabled();

    protected abstract String getStatsName();

    protected abstract Executor getCallbackExecutor();

    protected abstract boolean isScriptPresent(UUID var1);

    protected abstract boolean isExecEnabled(TenantId var1);

    protected abstract void reportExecution(TenantId var1, CustomerId var2);

    protected abstract ListenableFuture<UUID> doEvalScript(TenantId var1, ScriptType var2, String var3, UUID var4, String[] var5);

    protected abstract TbScriptExecutionTask doInvokeFunction(UUID var1, Object[] var2);

    protected abstract void doRelease(UUID var1) throws Exception;

    public void init() {
        if (this.getMaxEvalRequestsTimeout() > 0L || this.getMaxInvokeRequestsTimeout() > 0L) {
            this.timeoutExecutorService = ThingsBoardExecutors.newSingleThreadScheduledExecutor((String)"script-timeout");
        }
    }

    public void stop() {
        if (this.timeoutExecutorService != null) {
            this.timeoutExecutorService.shutdownNow();
        }
    }

    public void printStats() {
        if (this.isStatsEnabled()) {
            int pushed = this.pushedMsgs.getAndSet(0);
            int invoked = this.invokeMsgs.getAndSet(0);
            int evaluated = this.evalMsgs.getAndSet(0);
            int failed = this.failedMsgs.getAndSet(0);
            int timedOut = this.timeoutMsgs.getAndSet(0);
            if (pushed > 0 || invoked > 0 || evaluated > 0 || failed > 0 || timedOut > 0) {
                log.info("{}: pushed [{}] received [{}] invoke [{}] eval [{}] failed [{}] timedOut [{}]", new Object[]{this.getStatsName(), pushed, invoked + evaluated, invoked, evaluated, failed, timedOut});
            }
        }
    }

    @Override
    public ListenableFuture<UUID> eval(TenantId tenantId, ScriptType scriptType, String scriptBody, String ... argNames) {
        if (this.isExecEnabled(tenantId)) {
            if (this.scriptBodySizeExceeded(scriptBody)) {
                return this.error(String.format("Script body exceeds maximum allowed size of %s symbols", this.getMaxScriptBodySize()));
            }
            UUID scriptId = UUID.randomUUID();
            this.pushedMsgs.incrementAndGet();
            return this.withTimeoutAndStatsCallback(scriptId, null, this.doEvalScript(tenantId, scriptType, scriptBody, scriptId, argNames), this.evalCallback, this.getMaxEvalRequestsTimeout());
        }
        return this.error("Script Execution is disabled due to API limits!");
    }

    @Override
    public ListenableFuture<Object> invokeScript(TenantId tenantId, CustomerId customerId, UUID scriptId, Object ... args) {
        if (this.isExecEnabled(tenantId)) {
            if (!this.isScriptPresent(scriptId)) {
                return this.error("No compiled script found for scriptId: [" + scriptId + "]!");
            }
            if (!this.isDisabled(scriptId)) {
                if (this.argsSizeExceeded(args)) {
                    TbScriptException t = new TbScriptException(scriptId, TbScriptException.ErrorCode.OTHER, null, new IllegalArgumentException(String.format("Script input arguments exceed maximum allowed total args size of %s symbols", this.getMaxTotalArgsSize())));
                    return Futures.immediateFailedFuture((Throwable)this.handleScriptException(scriptId, null, t));
                }
                this.reportExecution(tenantId, customerId);
                this.pushedMsgs.incrementAndGet();
                log.trace("[{}] InvokeScript uuid {} with timeout {}ms", new Object[]{tenantId, scriptId, this.getMaxInvokeRequestsTimeout()});
                TbScriptExecutionTask task = this.doInvokeFunction(scriptId, args);
                ListenableFuture resultFuture = Futures.transform(task.getResultFuture(), output -> {
                    String result = JacksonUtil.toString((Object)output);
                    if (this.resultSizeExceeded(result)) {
                        throw new TbScriptException(scriptId, TbScriptException.ErrorCode.OTHER, null, new RuntimeException(String.format("Script invocation result exceeds maximum allowed size of %s symbols", this.getMaxResultSize())));
                    }
                    return output;
                }, (Executor)MoreExecutors.directExecutor());
                return this.withTimeoutAndStatsCallback(scriptId, task, resultFuture, this.invokeCallback, this.getMaxInvokeRequestsTimeout());
            }
            String message = "Script invocation is blocked due to maximum error count " + this.getMaxErrors() + ", scriptId " + scriptId + "!";
            log.warn("[{}] " + message, (Object)tenantId);
            return this.error(message);
        }
        return this.error("Script execution is disabled due to API limits!");
    }

    private <T extends V, V> ListenableFuture<T> withTimeoutAndStatsCallback(UUID scriptId, TbScriptExecutionTask task, ListenableFuture<T> future, FutureCallback<V> statsCallback, long timeout) {
        if (timeout > 0L) {
            future = Futures.withTimeout(future, (long)timeout, (TimeUnit)TimeUnit.MILLISECONDS, (ScheduledExecutorService)this.timeoutExecutorService);
        }
        Futures.addCallback(future, statsCallback, (Executor)this.getCallbackExecutor());
        return Futures.catchingAsync((ListenableFuture)future, Exception.class, input -> Futures.immediateFailedFuture((Throwable)this.handleScriptException(scriptId, task, (Throwable)input)), (Executor)MoreExecutors.directExecutor());
    }

    private Throwable handleScriptException(UUID scriptId, TbScriptExecutionTask task, Throwable t) {
        boolean timeout;
        boolean bl = timeout = t instanceof TimeoutException || t.getCause() != null && t.getCause() instanceof TimeoutException;
        if (timeout && task != null) {
            task.stop();
        }
        boolean blockList = timeout;
        String scriptBody = null;
        if (t instanceof TbScriptException) {
            TbScriptException scriptException = (TbScriptException)t;
            scriptBody = scriptException.getBody();
            Throwable cause = scriptException.getCause();
            switch (scriptException.getErrorCode()) {
                case COMPILATION: {
                    log.debug("[{}] Failed to compile script: {}", new Object[]{scriptId, scriptException.getBody(), cause});
                    break;
                }
                case TIMEOUT: {
                    log.debug("[{}] Timeout to execute script: {}", new Object[]{scriptId, scriptException.getBody(), cause});
                    break;
                }
                case OTHER: 
                case RUNTIME: {
                    log.debug("[{}] Failed to execute script: {}", new Object[]{scriptId, scriptException.getBody(), cause});
                }
            }
            boolean bl2 = blockList = timeout || scriptException.getErrorCode() != TbScriptException.ErrorCode.RUNTIME;
        }
        if (blockList) {
            BlockedScriptInfo disableListInfo = this.disabledScripts.computeIfAbsent(scriptId, key -> new BlockedScriptInfo(this.getMaxBlackListDurationSec()));
            int counter = disableListInfo.incrementAndGet();
            if (log.isDebugEnabled()) {
                log.debug("Script has exception counter {} on disabledFunctions for id {}, exception {}, cause {}, scriptBody {}", new Object[]{counter, scriptId, t, t.getCause(), scriptBody});
            } else {
                log.warn("Script has exception counter {} on disabledFunctions for id {}, exception {}", new Object[]{counter, scriptId, t.getMessage()});
            }
        }
        if (timeout) {
            return new TimeoutException("Script timeout!");
        }
        return t;
    }

    @Override
    public ListenableFuture<Void> release(UUID scriptId) {
        if (this.isScriptPresent(scriptId)) {
            try {
                this.disabledScripts.remove(scriptId);
                this.doRelease(scriptId);
            }
            catch (Exception e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
        }
        return Futures.immediateFuture(null);
    }

    private boolean isDisabled(UUID scriptId) {
        BlockedScriptInfo errorCount = this.disabledScripts.get(scriptId);
        if (errorCount != null) {
            if (errorCount.getExpirationTime() <= System.currentTimeMillis()) {
                this.disabledScripts.remove(scriptId);
                return false;
            }
            return errorCount.get() >= this.getMaxErrors();
        }
        return false;
    }

    private boolean scriptBodySizeExceeded(String scriptBody) {
        if (this.getMaxScriptBodySize() <= 0L) {
            return false;
        }
        return (long)scriptBody.length() > this.getMaxScriptBodySize();
    }

    private boolean argsSizeExceeded(Object[] args) {
        if (this.getMaxTotalArgsSize() <= 0L) {
            return false;
        }
        long totalArgsSize = 0L;
        for (Object arg : args) {
            if (arg instanceof CharSequence) {
                totalArgsSize += (long)((CharSequence)arg).length();
                continue;
            }
            String str = JacksonUtil.toString((Object)arg);
            if (str == null) continue;
            totalArgsSize += (long)str.length();
        }
        return totalArgsSize > this.getMaxTotalArgsSize();
    }

    private boolean resultSizeExceeded(String result) {
        if (this.getMaxResultSize() <= 0L) {
            return false;
        }
        return result != null && (long)result.length() > this.getMaxResultSize();
    }

    private <T> ListenableFuture<T> error(String message) {
        return Futures.immediateFailedFuture((Throwable)new RuntimeException(message));
    }
}

