/**
 * Copyright © 2016-2025 The Thingsboard Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.thingsboard.server.common.data.kv;

import lombok.Data;
import org.thingsboard.server.common.data.query.TsValue;

import java.util.ArrayList;
import java.util.List;

@Data
public class ReadTsKvQueryResult {

    private final int queryId;
    // Holds the data list;
    private final List<TsKvEntry> data;
    // Holds the max ts of the records that match aggregation intervals (not the ts of the aggregation window, but the ts of the last record among all the intervals)
    private final long lastEntryTs;

    public TsValue[] toTsValues() {
        if (data != null && !data.isEmpty()) {
            List<TsValue> queryValues = new ArrayList<>();
            for (TsKvEntry v : data) {
                queryValues.add(v.toTsValue()); // TODO: add count here.
            }
            return queryValues.toArray(new TsValue[queryValues.size()]);
        } else {
            return new TsValue[0];
        }
    }

    public TsValue toTsValue(ReadTsKvQuery query) {
        if (data == null || data.isEmpty()) {
            if (Aggregation.SUM.equals(query.getAggregation()) || Aggregation.COUNT.equals(query.getAggregation())) {
                long ts = query.getStartTs() + (query.getEndTs() - query.getStartTs()) / 2;
                return new TsValue(ts, "0");
            } else {
                return TsValue.EMPTY;
            }
        }
        if (data.size() > 1) {
            throw new RuntimeException("Query Result has multiple data points!");
        }
        return data.get(0).toTsValue();
    }

}
