/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ScatteringByteChannel;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.apache.kafka.common.memory.MemoryPool;
import org.apache.kafka.common.network.InvalidReceiveException;
import org.apache.kafka.common.network.Receive;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.thingsboard.server.common.data.exception.ThingsboardKafkaClientError;

public class NetworkReceive
implements Receive {
    public static final String UNKNOWN_SOURCE = "";
    public static final int UNLIMITED = -1;
    public static final int TB_MAX_REQUESTED_BUFFER_SIZE = 0x6400000;
    public static final int TB_LOG_REQUESTED_BUFFER_SIZE = 0xA00000;
    private static final Logger log = LoggerFactory.getLogger(NetworkReceive.class);
    private static final ByteBuffer EMPTY_BUFFER = ByteBuffer.allocate(0);
    private final String source;
    private final ByteBuffer size;
    private final int maxSize;
    private final MemoryPool memoryPool;
    private int requestedBufferSize = -1;
    private ByteBuffer buffer;

    public NetworkReceive(String source, ByteBuffer buffer) {
        this(0x6400000, source);
        this.buffer = buffer;
    }

    public NetworkReceive(String source) {
        this(0x6400000, source);
    }

    public NetworkReceive(int maxSize, String source) {
        this(maxSize, source, MemoryPool.NONE);
    }

    public NetworkReceive(int maxSize, String source, MemoryPool memoryPool) {
        this.source = source;
        this.size = ByteBuffer.allocate(4);
        this.buffer = null;
        this.maxSize = this.getMaxSize(maxSize);
        this.memoryPool = memoryPool;
    }

    public NetworkReceive() {
        this(UNKNOWN_SOURCE);
    }

    public String source() {
        return this.source;
    }

    public boolean complete() {
        return !this.size.hasRemaining() && this.buffer != null && !this.buffer.hasRemaining();
    }

    public long readFrom(ScatteringByteChannel channel) throws IOException {
        int bytesRead;
        int read = 0;
        if (this.size.hasRemaining()) {
            bytesRead = channel.read(this.size);
            if (bytesRead < 0) {
                throw new EOFException();
            }
            read += bytesRead;
            if (!this.size.hasRemaining()) {
                this.size.rewind();
                int receiveSize = this.size.getInt();
                if (receiveSize < 0) {
                    throw new InvalidReceiveException("Invalid receive (size = " + receiveSize + ")");
                }
                if (this.maxSize != -1 && receiveSize > this.maxSize) {
                    throw new ThingsboardKafkaClientError("Invalid receive (size = " + receiveSize + " larger than " + this.maxSize + ")");
                }
                this.requestedBufferSize = receiveSize;
                if (receiveSize == 0) {
                    this.buffer = EMPTY_BUFFER;
                }
            }
        }
        if (this.buffer == null && this.requestedBufferSize != -1) {
            if (this.requestedBufferSize > 0xA00000) {
                String stackTrace = Arrays.stream(Thread.currentThread().getStackTrace()).map(StackTraceElement::toString).collect(Collectors.joining("|"));
                log.error("Allocating buffer of size {} for source {}", (Object)this.requestedBufferSize, (Object)this.source);
                log.error("Stack Trace: {}", (Object)stackTrace);
            }
            this.buffer = this.memoryPool.tryAllocate(this.requestedBufferSize);
            if (this.buffer == null) {
                log.trace("Broker low on memory - could not allocate buffer of size {} for source {}", (Object)this.requestedBufferSize, (Object)this.source);
            }
        }
        if (this.buffer != null) {
            bytesRead = channel.read(this.buffer);
            if (bytesRead < 0) {
                throw new EOFException();
            }
            read += bytesRead;
        }
        return read;
    }

    public boolean requiredMemoryAmountKnown() {
        return this.requestedBufferSize != -1;
    }

    public boolean memoryAllocated() {
        return this.buffer != null;
    }

    public void close() throws IOException {
        if (this.buffer != null && this.buffer != EMPTY_BUFFER) {
            this.memoryPool.release(this.buffer);
            this.buffer = null;
        }
    }

    public ByteBuffer payload() {
        return this.buffer;
    }

    public int bytesRead() {
        if (this.buffer == null) {
            return this.size.position();
        }
        return this.buffer.position() + this.size.position();
    }

    public int size() {
        return this.payload().limit() + this.size.limit();
    }

    private int getMaxSize(int maxSize) {
        return maxSize == -1 ? 0x6400000 : Math.min(maxSize, 0x6400000);
    }
}

